/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file joiner.cpp
 *
 * Takes evaluated fragments (energy and forces) and by reading the factors files determines total energy
 * and each force for the whole molecule.
 *
 */

//============================ INCLUDES ===========================

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cstring>
#include <cmath>

#include "datacreator.hpp"
#include "Element/periodentafel.hpp"
#include "Fragmentation/defs.hpp"
#include "Fragmentation/EnergyMatrix.hpp"
#include "Fragmentation/ForceMatrix.hpp"
#include "Fragmentation/helpers.hpp"
#include "Fragmentation/HessianMatrix.hpp"
#include "Fragmentation/KeySetsContainer.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

//============================== MAIN =============================

int main(int argc, char **argv)
{
  periodentafel *periode = NULL; // and a period table of all elements
  EnergyMatrix Energy;
  EnergyMatrix EnergyFragments;
  
  EnergyMatrix Hcorrection;
  EnergyMatrix HcorrectionFragments;
  
  ForceMatrix Force;
  ForceMatrix ForceFragments;

  HessianMatrix Hessian;
  HessianMatrix HessianFragments;
  
  ForceMatrix Shielding;
  ForceMatrix ShieldingPAS;
  ForceMatrix ShieldingFragments;
  ForceMatrix ShieldingPASFragments;
  ForceMatrix Chi;
  ForceMatrix ChiPAS;
  ForceMatrix ChiFragments;
  ForceMatrix ChiPASFragments;
  KeySetsContainer KeySet;  
  stringstream prefix;
  char *dir = NULL;
  bool NoHCorrection = false;
  bool NoHessian = false;

  LOG(0, "Joiner");
  LOG(0, "======");

  // Get the command line options
  if (argc < 3) {
    LOG(0, "Usage: " << argv[0] << " <inputdir> <prefix> [elementsdb]");
    LOG(0, "<inputdir>\ttherein the output of a molecuilder fragmentation is expected, each fragment with a subdir containing an energy.all and a forces.all file.");
    LOG(0, "<prefix>\tprefix of energy and forces file.");
    LOG(0, "[elementsdb]\tpath to elements database, needed for shieldings.");
    return 1;
  } else {
    dir = new char[strlen(argv[2]) + 2];
    strcpy(dir, "/");
    strcat(dir, argv[2]);
  }
  if (argc > 3) {
    periode = new periodentafel;
    periode->LoadPeriodentafel(argv[3]);
  }

  // Test the given directory
  if (!TestParams(argc, argv))
    return 1;

  // +++++++++++++++++ PARSING +++++++++++++++++++++++++++++++

  // ------------- Parse through all Fragment subdirs --------
  if (!Energy.ParseFragmentMatrix(argv[1], dir, EnergySuffix, 0,0)) return 1;
  if (!Hcorrection.ParseFragmentMatrix(argv[1], "", HCORRECTIONSUFFIX, 0,0)) {
    NoHCorrection = true;
    LOG(0, "No HCorrection matrices found, skipping these.");
  }
  if (!Force.ParseFragmentMatrix(argv[1], dir, ForcesSuffix, 0,0)) return 1;
  if (!Hessian.ParseFragmentMatrix(argv[1], dir, HessianSuffix, 0,0)) {
    NoHessian = true;
    LOG(0, "No hessian matrices found, skipping these.");
  }
  if (periode != NULL) { // also look for PAS values
    if (!Shielding.ParseFragmentMatrix(argv[1], dir, ShieldingSuffix, 1, 0)) return 1;
    if (!ShieldingPAS.ParseFragmentMatrix(argv[1], dir, ShieldingPASSuffix, 1, 0)) return 1;
    if (!Chi.ParseFragmentMatrix(argv[1], dir, ChiSuffix, 1, 0)) return 1;
    if (!ChiPAS.ParseFragmentMatrix(argv[1], dir, ChiPASSuffix, 1, 0)) return 1;
  }

  // ---------- Parse the TE Factors into an array -----------------
  if (!Energy.InitialiseIndices()) return 1;
  if (!NoHCorrection) 
    Hcorrection.InitialiseIndices();
  
  // ---------- Parse the Force indices into an array ---------------
  if (!Force.ParseIndices(argv[1])) return 1;

  // ---------- Parse the Hessian (=force) indices into an array ---------------
  if (!NoHessian)
    if (!Hessian.InitialiseIndices((class MatrixContainer *)&Force)) return 1;

  // ---------- Parse the shielding indices into an array ---------------
  if (periode != NULL) { // also look for PAS values
    if(!Shielding.ParseIndices(argv[1])) return 1;
    if(!ShieldingPAS.ParseIndices(argv[1])) return 1;
    if(!Chi.ParseIndices(argv[1])) return 1;
    if(!ChiPAS.ParseIndices(argv[1])) return 1;
  }

  // ---------- Parse the KeySets into an array ---------------
  if (!KeySet.ParseKeySets(argv[1], Force.RowCounter, Force.MatrixCounter)) return 1;

  if (!KeySet.ParseManyBodyTerms()) return 1;

  if (!EnergyFragments.AllocateMatrix(Energy.Header, Energy.MatrixCounter, Energy.RowCounter, Energy.ColumnCounter)) return 1;
  if (!NoHCorrection)  
    HcorrectionFragments.AllocateMatrix(Hcorrection.Header, Hcorrection.MatrixCounter, Hcorrection.RowCounter, Hcorrection.ColumnCounter);
  if (!ForceFragments.AllocateMatrix(Force.Header, Force.MatrixCounter, Force.RowCounter, Force.ColumnCounter)) return 1;
  if (!NoHessian)
    if (!HessianFragments.AllocateMatrix(Hessian.Header, Hessian.MatrixCounter, Hessian.RowCounter, Hessian.ColumnCounter)) return 1;
  if (periode != NULL) { // also look for PAS values
    if (!ShieldingFragments.AllocateMatrix(Shielding.Header, Shielding.MatrixCounter, Shielding.RowCounter, Shielding.ColumnCounter)) return 1;
    if (!ShieldingPASFragments.AllocateMatrix(ShieldingPAS.Header, ShieldingPAS.MatrixCounter, ShieldingPAS.RowCounter, ShieldingPAS.ColumnCounter)) return 1;
    if (!ChiFragments.AllocateMatrix(Chi.Header, Chi.MatrixCounter, Chi.RowCounter, Chi.ColumnCounter)) return 1;
    if (!ChiPASFragments.AllocateMatrix(ChiPAS.Header, ChiPAS.MatrixCounter, ChiPAS.RowCounter, ChiPAS.ColumnCounter)) return 1;
  }

  // ----------- Resetting last matrices (where full QM values are stored right now)
  if(!Energy.SetLastMatrix(0., 0)) return 1;
  if(!Force.SetLastMatrix(0., 2)) return 1;
  if (!NoHessian)
    if (!Hessian.SetLastMatrix(0., 0)) return 1;
  if (periode != NULL) { // also look for PAS values
    if(!Shielding.SetLastMatrix(0., 2)) return 1;
    if(!ShieldingPAS.SetLastMatrix(0., 2)) return 1;
    if(!Chi.SetLastMatrix(0., 2)) return 1;
    if(!ChiPAS.SetLastMatrix(0., 2)) return 1;
  }

  // +++++++++++++++++ SUMMING +++++++++++++++++++++++++++++++

  // --------- sum up and write for each order----------------
  for (int BondOrder=0;BondOrder<KeySet.Order;BondOrder++) {
    // --------- sum up energy --------------------
    LOG(0, "Summing energy of order " << BondOrder+1 << " ...");
    if (!EnergyFragments.SumSubManyBodyTerms(Energy, KeySet, BondOrder)) return 1;
    if (!NoHCorrection) { 
      HcorrectionFragments.SumSubManyBodyTerms(Hcorrection, KeySet, BondOrder);
      if (!Energy.SumSubEnergy(EnergyFragments, &HcorrectionFragments, KeySet, BondOrder, 1.)) return 1;
      Hcorrection.SumSubEnergy(HcorrectionFragments, NULL, KeySet, BondOrder, 1.);
    } else 
      if (!Energy.SumSubEnergy(EnergyFragments, NULL, KeySet, BondOrder, 1.)) return 1;
    // --------- sum up Forces --------------------
    LOG(0, "Summing forces of order " << BondOrder+1 << " ...");
    if (!ForceFragments.SumSubManyBodyTerms(Force, KeySet, BondOrder)) return 1;
    if (!Force.SumSubForces(ForceFragments, KeySet, BondOrder, 1.)) return 1;
    // --------- sum up Hessian --------------------
    if (!NoHessian) {
      LOG(0, "Summing Hessian of order " << BondOrder+1 << " ...");
      if (!HessianFragments.SumSubManyBodyTerms(Hessian, KeySet, BondOrder)) return 1;
      if (!Hessian.SumSubHessians(HessianFragments, KeySet, BondOrder, 1.)) return 1;
    }
    if (periode != NULL) { // also look for PAS values
      LOG(0, "Summing shieldings and susceptibilities of order " << BondOrder+1 << " ...");
      if (!ShieldingFragments.SumSubManyBodyTerms(Shielding, KeySet, BondOrder)) return 1;
      if (!Shielding.SumSubForces(ShieldingFragments, KeySet, BondOrder, 1.)) return 1;
      if (!ShieldingPASFragments.SumSubManyBodyTerms(ShieldingPAS, KeySet, BondOrder)) return 1;
      if (!ShieldingPAS.SumSubForces(ShieldingPASFragments, KeySet, BondOrder, 1.)) return 1;
      if (!ChiFragments.SumSubManyBodyTerms(Chi, KeySet, BondOrder)) return 1;
      if (!Chi.SumSubForces(ChiFragments, KeySet, BondOrder, 1.)) return 1;
      if (!ChiPASFragments.SumSubManyBodyTerms(ChiPAS, KeySet, BondOrder)) return 1;
      if (!ChiPAS.SumSubForces(ChiPASFragments, KeySet, BondOrder, 1.)) return 1;
    }

    // --------- write the energy and forces file --------------------
    prefix.str(" ");
    prefix << dir << OrderSuffix << (BondOrder+1);
    LOG(0, "Writing files " << argv[1] << prefix.str() << ". ...");
    // energy
    if (!Energy.WriteLastMatrix(argv[1], (prefix.str()).c_str(), EnergySuffix)) return 1;
    // forces
    if (!Force.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ForcesSuffix)) return 1;
    // hessian
    if (!NoHessian)
      if (!Hessian.WriteLastMatrix(argv[1], (prefix.str()).c_str(), HessianSuffix)) return 1;
    // shieldings
    if (periode != NULL) { // also look for PAS values
      if (!Shielding.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ShieldingSuffix)) return 1;
      if (!ShieldingPAS.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ShieldingPASSuffix)) return 1;
      if (!Chi.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ChiSuffix)) return 1;
      if (!ChiPAS.WriteLastMatrix(argv[1], (prefix.str()).c_str(), ChiPASSuffix)) return 1;
    }
  }
  // fragments
  prefix.str(" ");
  prefix << dir << EnergyFragmentSuffix;
  if (!EnergyFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  if (!NoHCorrection) {
    prefix.str(" ");
    prefix << dir << HcorrectionFragmentSuffix;
    if (!HcorrectionFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  }
  prefix.str(" ");
  prefix << dir << ForceFragmentSuffix;
  if (!ForceFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  {
    std::string fileprefix(FRAGMENTPREFIX);
    fileprefix += ENERGYPERFRAGMENT;
    if (!CreateDataFragment(EnergyFragments, KeySet, argv[1], fileprefix.c_str(), "fragment energy versus the Fragment No", "today", CreateEnergy)) return 1;
  }
  if (!NoHessian) {
    prefix.str(" ");
    prefix << dir << HessianFragmentSuffix;
    if (!HessianFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  }
  if (periode != NULL) { // also look for PAS values
    prefix.str(" ");
    prefix << dir << ShieldingFragmentSuffix;
    if (!ShieldingFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
    prefix.str(" ");
    prefix << dir << ShieldingPASFragmentSuffix;
    if (!ShieldingPASFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
    prefix.str(" ");
    prefix << dir << ChiFragmentSuffix;
    if (!ChiFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
    prefix.str(" ");
    prefix << dir << ChiPASFragmentSuffix;
    if (!ChiPASFragments.WriteTotalFragments(argv[1], (prefix.str()).c_str())) return 1;
  }

  // write last matrices as fragments into central dir (not subdir as above), for analyzer to know index bounds
  if (!Energy.WriteLastMatrix(argv[1], dir, EnergyFragmentSuffix)) return 1;
  if (!NoHCorrection) Hcorrection.WriteLastMatrix(argv[1], dir, HcorrectionFragmentSuffix);
  if (!Force.WriteLastMatrix(argv[1], dir, ForceFragmentSuffix)) return 1;
  if (!NoHessian)
    if (!Hessian.WriteLastMatrix(argv[1], dir, HessianFragmentSuffix)) return 1;
  if (periode != NULL) { // also look for PAS values
    if (!Shielding.WriteLastMatrix(argv[1], dir, ShieldingFragmentSuffix)) return 1;
    if (!ShieldingPAS.WriteLastMatrix(argv[1], dir, ShieldingPASFragmentSuffix)) return 1;
    if (!Chi.WriteLastMatrix(argv[1], dir, ChiFragmentSuffix)) return 1;
    if (!ChiPAS.WriteLastMatrix(argv[1], dir, ChiPASFragmentSuffix)) return 1;
  }

  // exit
  delete(periode);
  delete[](dir);
  LOG(0, "done.");
  return 0;
};

//============================ END ===========================
