/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CommandLineStatusIndicator.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"
#include "CodePatterns/Observer/Notification.hpp"
#include "CodePatterns/Observer/Observable.hpp"

#include "CommandLineUI/CommandLineStatusIndicator.hpp"

#include <iostream>

#include "Actions/ActionQueue.hpp"
#include "Actions/ActionStatusList.hpp"
#include "Actions/Process.hpp"

using namespace MoleCuilder;

CommandLineStatusIndicator::CommandLineStatusIndicator() :
    Observer("CommandLineStatusIndicator"),
    StatusList(ActionQueue::getInstance().getStatusList()),
    StatusList_signedOn(false)
{
  Process::AddObserver(this);

  StatusList.signOn(this, ActionStatusList::StatusAdded);
  StatusList_signedOn = true;
}

CommandLineStatusIndicator::~CommandLineStatusIndicator()
{
  Process::RemoveObserver(this);

  if (StatusList_signedOn) {
    // print all remaining messages prior to exit
    while (StatusList.size() != 0)
      displayStatusMessage();
    StatusList.signOff(this, ActionStatusList::StatusAdded);
  }
}

void CommandLineStatusIndicator::update(Observable *subject){
  if (subject == &StatusList) {
    // we do not react to general updates from StatusList
  } else {
    Process *proc;
    // we are only observing Processes
    if((proc=dynamic_cast<Process*>(subject))){
      // see what kind of progress we have to display
      if(proc->getMaxSteps()>0){
        if(!proc->doesStop()){
          std::cout << "Busy (" << proc->getName() << ") ";
          // we can show a percentage done
          std::cout << (int)proc->getDoneRatio() << "% done" << std::endl;
        }
      }
      else{
        // we only show some kind of busy animation
        if(proc->doesStart()){
          std::cout << "Busy (" << proc->getName() << ")";
        }
        if(!proc->doesStop()){
          std::cout << ".";
        }
        else {
          std::cout << std::endl;
        }
      }
    }
  }
}

void CommandLineStatusIndicator::subjectKilled(Observable *subject)
{
  if (subject == &StatusList) {
    // print all remaining messages
    while (StatusList.size() != 0)
      displayStatusMessage();
    // don't need to sign off, just note down that we are
    StatusList_signedOn = false;
  }
}

void CommandLineStatusIndicator::displayStatusMessage() const
{
  const std::string message = StatusList.popFirstMessage();
  std::cout << message << std::endl;
}

void CommandLineStatusIndicator::recieveNotification(Observable *_publisher, Notification *_notification)
{
  if (_publisher == &StatusList) {
    switch(_notification->getChannelNo()) {
      case ActionStatusList::StatusAdded:
        displayStatusMessage();
        break;
    }
  }
}

