/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * SubgraphDissectionAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"

#include "atom.hpp"
#include "Bond/bond.hpp"
#include "config.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Graph/BondGraph.hpp"
#include "Graph/DepthFirstSearchAnalysis.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

typedef std::map< moleculeId_t, std::vector<atomId_t> > MolAtomList;
typedef std::map< atomId_t, atomId_t > AtomAtomList;

#include "Actions/GraphAction/SubgraphDissectionAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "SubgraphDissectionAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr GraphSubgraphDissectionAction::performCall() {
  // obtain information
  getParametersfromValueStorage();


  // first create stuff for undo state
  LOG(0, "STATUS: Creating undo state.");
  MolAtomList moleculelist;
  vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  for (vector<molecule *>::const_iterator moliter = allmolecules.begin(); moliter != allmolecules.end(); ++moliter) {
    std::vector<atomId_t> atomlist;
    atomlist.resize((*moliter)->size());
    for (molecule::const_iterator atomiter = (*moliter)->begin(); atomiter != (*moliter)->end(); ++atomiter) {
      atomlist.push_back((*atomiter)->getId());
    }
    moleculelist.insert( std::pair< moleculeId_t, std::vector<atomId_t> > ((*moliter)->getId(), atomlist));
  }
  GraphSubgraphDissectionState *UndoState = new GraphSubgraphDissectionState(moleculelist, params);

  // 0a. remove all present molecules
  LOG(0, "STATUS: Removing all present molecules.");
  MoleculeListClass *molecules = World::getInstance().getMolecules();
  for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
    molecules->erase(*MolRunner);
    World::getInstance().destroyMolecule(*MolRunner);
  }

  // 0b. remove all bonds and construct a molecule with all atoms
  molecule *mol = World::getInstance().createMolecule();
  {
    vector <atom *> allatoms = World::getInstance().getAllAtoms();
    for(vector<atom *>::iterator AtomRunner = allatoms.begin(); AtomRunner != allatoms.end(); ++AtomRunner) {
//      const BondList& ListOfBonds = (*AtomRunner)->getListOfBonds();
//      for(BondList::iterator BondRunner = ListOfBonds.begin();
//          !ListOfBonds.empty();
//          BondRunner = ListOfBonds.begin()) {
//        delete(*BondRunner);
//      }
      mol->AddAtom(*AtomRunner);
    }
  }

  // 1. create the bond structure of the single molecule
  LOG(0, "STATUS: (Re-)constructing adjacency.");
  if (mol->getBondCount() == 0) {
    BondGraph *BG = World::getInstance().getBondGraph();
    molecule::atomVector Set = mol->getAtomSet();
    BG->CreateAdjacency(Set);
//    if (mol->getBondCount() == 0) {
//      World::getInstance().destroyMolecule(mol);
//      ELOG(1, "There are no bonds.");
//      return Action::failure;
//    }
  }

  // 2. scan for connected subgraphs
  DepthFirstSearchAnalysis DFS;
  DFS();
  DFS.UpdateMoleculeStructure();
  if (!World::getInstance().numMolecules()) {
    //World::getInstance().destroyMolecule(mol);
    DoeLog(1) && (eLog()<< Verbose(1) << "There are no atoms." << endl);
    return Action::failure;
  }

  DoLog(1) && (Log() << Verbose(1) << "I scanned " << World::getInstance().numMolecules() << " molecules." << endl);

  return Action::state_ptr(UndoState);
}

Action::state_ptr GraphSubgraphDissectionAction::performUndo(Action::state_ptr _state) {
  GraphSubgraphDissectionState *state = assert_cast<GraphSubgraphDissectionState*>(_state.get());

  {
    // remove all present molecules
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
    for (vector<molecule *>::iterator MolRunner = allmolecules.begin(); MolRunner != allmolecules.end(); ++MolRunner) {
      molecules->erase(*MolRunner);
      World::getInstance().destroyMolecule(*MolRunner);
    }
  }

  {
    // construct the old state
    MoleculeListClass *molecules = World::getInstance().getMolecules();
    molecule *mol = NULL;
    for (MolAtomList::const_iterator iter = state->moleculelist.begin(); iter != state->moleculelist.end(); ++iter) {
      mol = World::getInstance().createMolecule();
      if (mol->getId() != (*iter).first)
        World::getInstance().changeMoleculeId(mol->getId(), (*iter).first);
      for (std::vector<atomId_t>::const_iterator atomiter = (*iter).second.begin(); atomiter != (*iter).second.end(); ++atomiter) {
        atom *Walker = World::getInstance().getAtom(AtomById(*atomiter));
        mol->AddAtom(Walker);
      }
      molecules->insert(mol);
    }
  }

  return Action::state_ptr(_state);
}

Action::state_ptr GraphSubgraphDissectionAction::performRedo(Action::state_ptr _state){
  return performCall();
}

bool GraphSubgraphDissectionAction::canUndo() {
  return true;
}

bool GraphSubgraphDissectionAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
