/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * InterfaceVMGJob.cpp
 *
 *  Created on: 10.06.2012
 *      Author: Frederik Heber
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MPI
#include "mpi.h"
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cmath>
#include <iostream>

#include "CodePatterns/Log.hpp"

#include "base/vector.hpp"
#include "base/math.hpp"
#ifdef HAVE_MPI
#include "comm/comm_mpi.hpp"
#else
#include "comm/comm_serial.hpp"
#endif
#include "grid/grid.hpp"
#include "grid/multigrid.hpp"

#include "InterfaceVMGJob.hpp"

using namespace VMG;
using VMGInterfaces::InterfaceVMGJob;

InterfaceVMGJob::InterfaceVMGJob(const std::vector< double > &_sampled_input,
    std::vector< double > &_sampled_output,
    const std::vector< std::vector<double> > &_particle_positions,
    const std::vector< double > &_particle_charges,
    VMG::Boundary boundary,
    int levelMin,
    int levelMax,
    const VMG::Vector &box_begin,
    vmg_float box_end,
    const int& near_field_cells,
    int coarseningSteps,
    double alpha) :
  VMG::Interface(boundary, levelMin, levelMax,
      box_begin, box_end, coarseningSteps, alpha),
  spl(near_field_cells, Extent(MaxLevel()).MeshWidth().Max()),
  sampled_input(_sampled_input),
  sampled_output(_sampled_output),
  level(levelMax)
{
  std::vector< std::vector<double> >::const_iterator positer = _particle_positions.begin();
  std::vector<double>::const_iterator chargeiter = _particle_charges.begin();
  double pos[3];
  for (; positer != _particle_positions.end(); ++positer, ++chargeiter) {
    ASSERT( (*positer).size() == 3,
        "InterfaceVMGJob::InterfaceVMGJob() - particle "
        +toString(distance(_particle_positions.begin(), positer))+" has not exactly 3 coordinates.");
    for (size_t i=0;i<3;++i)
      pos[i] = (*positer)[i];
    particles.push_back(Particle::Particle(pos, *chargeiter));
  }
}

InterfaceVMGJob::~InterfaceVMGJob()
{}

void InterfaceVMGJob::ImportRightHandSide(Multigrid& multigrid)
{
  Index i;
  Vector pos;
  //  VMG::TempGrid *temp_grid = new VMG::TempGrid(129, 0, 0., 1.);

  Grid& grid = multigrid(multigrid.MaxLevel());
  grid.ClearBoundary();

  /// 1. assign nuclei as smeared-out charges to the grid

  /*
   * Charge assignment on the grid
   */
#ifdef HAVE_MPI
  CommMPI& comm = *dynamic_cast<CommMPI*>(VMG::MG::GetComm());
#else
  CommSerial& comm = *dynamic_cast<CommSerial*>(VMG::MG::GetComm());
#endif
  Grid& particle_grid = comm.GetParticleGrid();

  particle_grid.Clear();

  assert(particle_grid.Global().LocalSize().IsComponentwiseGreater(
      VMG::MG::GetFactory().GetObjectStorageVal<int>("PARTICLE_NEAR_FIELD_CELLS")));

  for (std::list<Particle::Particle>::iterator iter = particles.begin();
      iter != particles.end(); ++iter)
    spl.SetSpline(particle_grid, *iter);

  // Communicate charges over halo
  comm.CommFromGhosts(particle_grid);

  // Assign charge values to the right hand side
  for (int i=0; i<grid.Local().Size().X(); ++i)
    for (int j=0; j<grid.Local().Size().Y(); ++j)
      for (int k=0; k<grid.Local().Size().Z(); ++k)
  grid(grid.Local().Begin().X() + i,
       grid.Local().Begin().Y() + j,
       grid.Local().Begin().Z() + k) = 4.0 * VMG::Math::pi *
    particle_grid.GetVal(particle_grid.Local().Begin().X() + i,
             particle_grid.Local().Begin().Y() + j,
             particle_grid.Local().Begin().Z() + k);

  /// 2. add sampled electron density to the grid

  const Index begin_local = grid.Global().LocalBegin() - grid.Local().HaloSize1();

  LOG(1, "INFO: Mesh has extent " << grid.Extent().MeshWidth() << ".");
  const int gridpoints = pow(2, level);
  LOG(1, "INFO: gridpoints on finest level are " << gridpoints << ".");
  assert( (grid.Extent().MeshWidth().X() * gridpoints) == 1 );
  assert( (grid.Extent().MeshWidth().Y() * gridpoints) == 1 );
  assert( (grid.Extent().MeshWidth().Z() * gridpoints) == 1 );
  LOG(1, "INFO: "
      << "X in [" << grid.Local().Begin().X() << "," << grid.Local().End().X() << "],"
      << "Y in [" << grid.Local().Begin().Y() << "," << grid.Local().End().Y() << "],"
      << "Z in [" << grid.Local().Begin().Z() << "," << grid.Local().End().Z() << "].");

  const double element_volume =
      grid.Extent().MeshWidth().X() * grid.Extent().MeshWidth().Y() * grid.Extent().MeshWidth().Z();
  std::vector<double>::const_iterator sample_iter = sampled_input.begin();
  for (i.X()=grid.Local().Begin().X(); i.X()<grid.Local().End().X(); ++i.X())
    for (i.Y()=grid.Local().Begin().Y(); i.Y()<grid.Local().End().Y(); ++i.Y())
      for (i.Z()=grid.Local().Begin().Z(); i.Z()<grid.Local().End().Z(); ++i.Z())
        grid(i) -= (*sample_iter++) * element_volume; //temp_grid(i);
  assert( sample_iter == sampled_input.end() );

  Grid::iterator grid_iter;
  double charge_sum = 0.0;
  for (grid_iter=grid.Iterators().Local().Begin(); grid_iter!=grid.Iterators().Local().End(); ++grid_iter)
    charge_sum += grid.GetVal(*grid_iter);
//  charge_sum = MG::GetComm()->GlobalSum(charge_sum);
//  comm.PrintStringOnce("Grid charge sum: %e", charge_sum);
  std::cout << "Grid charge sum: " << charge_sum << std::endl;

//  delete temp_grid;
}

void InterfaceVMGJob::ExportSolution(Grid& grid)
{
  // grid now contains the sough-for potential

  const Index begin_local = grid.Global().LocalBegin() - grid.Local().HaloSize1();
  Index i;

  sampled_output.clear();
  for (i.X()=grid.Local().Begin().X(); i.X()<grid.Local().End().X(); ++i.X())
    for (i.Y()=grid.Local().Begin().Y(); i.Y()<grid.Local().End().Y(); ++i.Y())
      for (i.Z()=grid.Local().Begin().Z(); i.Z()<grid.Local().End().Z(); ++i.Z()) {
        sampled_output.push_back(grid(i));
      }

  Grid::iterator grid_iter;
  double potential_sum = 0.0;
  for (grid_iter=grid.Iterators().Local().Begin(); grid_iter!=grid.Iterators().Local().End(); ++grid_iter)
    potential_sum += grid.GetVal(*grid_iter);
//  charge_sum = MG::GetComm()->GlobalSum(potential_sum);
//  comm.PrintStringOnce("Grid potential sum: %e", potential_sum);
  std::cout << "Grid potential sum: " << potential_sum << std::endl;

  //Particle::CommMPI& comm = *dynamic_cast<Particle::CommMPI*>(MG::GetComm());
  //e_long = comm.GlobalSumRoot(e_long);

}
