/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * XyzParser.cpp
 *
 *  Created on: Mar 2, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "XyzParser.hpp"
#include "World.hpp"
#include "atom.hpp"
#include "molecule.hpp"
#include "element.hpp"
#include "periodentafel.hpp"

using namespace std;

/**
 * Constructor.
 */
XyzParser::XyzParser() :
  comment("")
{}

/**
 * Destructor.
 */
XyzParser::~XyzParser() {
}

/**
 * Loads an XYZ file into the World.
 *
 * \param XYZ file
 */
void XyzParser::load(istream* file) {
  atom* newAtom = NULL;
  molecule* newmol = NULL;
  int numberOfAtoms;
  char commentBuffer[512], type[3];
  double tmp;

  // the first line tells number of atoms, the second line is always a comment
  *file >> numberOfAtoms >> ws;
  file->getline(commentBuffer, 512);
  comment = commentBuffer;

  newmol = World::getInstance().createMolecule();
  newmol->ActiveFlag = true;
  // TODO: Remove the insertion into molecule when saving does not depend on them anymore. Also, remove molecule.hpp include
  World::getInstance().getMolecules()->insert(newmol);
  for (int i = 0; i < numberOfAtoms; i++) {
    newAtom = World::getInstance().createAtom();
    *file >> type;
    for (int j=0;j<NDIM;j++) {
      *file >> tmp;
      newAtom->set(j, tmp);
    }
    newAtom->setType(World::getInstance().getPeriode()->FindElement(type));
    newmol->AddAtom(newAtom);
  }
}

/**
 * Saves the \a atoms into as a XYZ file.
 *
 * \param file where to save the state
 * \param atoms atoms to store
 */
void XyzParser::save(ostream* file, const std::vector<atom *> &atoms) {
  DoLog(0) && (Log() << Verbose(0) << "Saving changes to xyz." << std::endl);
  //if (comment == "") {
    time_t now = time((time_t *)NULL);   // Get the system time and put it into 'now' as 'calender time'
    comment = "Created by molecuilder on ";
    // ctime ends in \n\0, we have to cut away the newline
    std::string time(ctime(&now));
    size_t pos = time.find('\n');
    if (pos != 0)
      comment += time.substr(0,pos);
    else
      comment += time;
  //}
  *file << atoms.size() << endl << "\t" << comment << endl;

  for(vector<atom*>::const_iterator it = atoms.begin(); it != atoms.end(); it++) {
    *file << noshowpoint << (*it)->getType()->getSymbol() << "\t" << (*it)->at(0) << "\t" << (*it)->at(1) << "\t" << (*it)->at(2) << endl;
  }
}
