/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * TextWindow.cpp
 *
 *  Created on: Jan 7, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>

#include "Menu/TextMenu/TextMenu.hpp"
#include "Menu/TextMenu/TxMenu.hpp"
#include "Menu/TextMenu/ActionMenuItem.hpp"
#include "Menu/TextMenu/SeparatorMenuItem.hpp"
#include "Menu/TextMenu/DisplayMenuItem.hpp"
#include "Menu/TextMenu/SubMenuItem.hpp"
#include "TextUI/TextStatusIndicator.hpp"
#include "TextUI/TextWindow.hpp"
#include "Actions/MethodAction.hpp"
#include "Actions/ErrorAction.hpp"
#include "Actions/ActionQueue.hpp"
#include "Actions/ActionTrait.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "molecule.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Views/StreamStringView.hpp"
#include "Views/MethodStringView.hpp"
#include "World.hpp"

#include "Helpers/defs.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include <iostream>
#include <map>

using namespace MoleCuilder;

// TODO: see what code can be moved to a base class for Graphic and Text Windows
TextWindow::TextWindow()
{
  // build the main menu
  main_menu = new TextMenu<TxMenu>(std::cout, "");
  // reserve s for save and q for quite
  main_menu->reserveShortcut('s', "output");
  main_menu->reserveShortcut('q', "quit");

  moleculeView = new StreamStringView(boost::bind(&TextWindow::Enumerate,this,_1));
  new DisplayMenuItem(main_menu->getMenuInstance(),moleculeView,"Molecule List");

  new SeparatorMenuItem(main_menu->getMenuInstance());

  main_menu->init();

  new SeparatorMenuItem(main_menu->getMenuInstance());

  // save has reserved key 's'
//  const Action &saveConfigAction = ActionQueue::getInstance().getActionByName("output");
  new ActionMenuItem('s',"save current setup to config files",main_menu->getMenuInstance(),"output");

  // create a specific quit action for this UI and store in registry
  ActionTrait quitTrait(OptionTrait("quit", &typeid(void), "quits the program"));
  quitAction = new MethodAction(quitTrait,boost::bind(&TxMenu::doQuit,main_menu->getMenuInstance()));
  MoleCuilder::ActionQueue::getInstance().registerAction(quitAction);
  new ActionMenuItem('q',"quit",main_menu->getMenuInstance(),"quit");

  // Add status indicators etc...
  statusIndicator = new TextStatusIndicator();
}

void TextWindow::Enumerate(std::ostream *out)
{
  periodentafel *periode = World::getInstance().getPeriode();
  std::vector<molecule *> allmolecules = World::getInstance().getAllMolecules();
  std::map<atomicNumber_t,unsigned int> counts;
  double size=0;
  Vector Origin;

  // header
  (*out) << "Index\tName\t\tAtoms\tFormula\tCenter\tSize" << endl;
  (*out) << "-----------------------------------------------" << endl;
  if (allmolecules.size() == 0)
    (*out) << "\tNone" << endl;
  else {
    Origin.Zero();
    for (std::vector<molecule *>::const_iterator ListRunner = allmolecules.begin();
        ListRunner != allmolecules.end(); ListRunner++) {
      // count atoms per element and determine size of bounding sphere
      size=0.;
      const molecule *MolRunner = *ListRunner;
      for (molecule::const_iterator iter = MolRunner->begin(); iter != MolRunner->end(); ++iter) {
        counts[(*iter)->getType()->getAtomicNumber()]++;
        if ((*iter)->DistanceSquared(Origin) > size)
          size = (*iter)->DistanceSquared(Origin);
      }
      // output Index, Name, number of atoms, chemical formula
      (*out) << (MolRunner->ActiveFlag ? "*" : " ") << MolRunner->IndexNr << "\t" << MolRunner->name << "\t\t" << MolRunner->getAtomCount() << "\t";

      std::map<atomicNumber_t,unsigned int>::reverse_iterator iter;
      for(iter=counts.rbegin(); iter!=counts.rend();++iter){
        atomicNumber_t Z =(*iter).first;
        (*out) << periode->FindElement(Z)->getSymbol() << (*iter).second;
      }
      // Center and size
      Vector Center = MolRunner->DetermineCenterOfAll();
      (*out) << "\t" << Center << "\t" << sqrt(size) << endl;
    }
  }
}

TextWindow::~TextWindow()
{
  delete moleculeView;
  delete statusIndicator;
  delete main_menu;
}

void TextWindow::display() {
  main_menu->display();
}

