/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ValueStorage.cpp
 *
 *  Created on: Jul 22, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Singleton_impl.hpp"

#include "Actions/OptionTrait.hpp"
#include "Actions/OptionRegistry.hpp"
#include "Actions/Values.hpp"
#include "Actions/ValueStorage.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "LinearAlgebra/BoxVector.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "atom.hpp"
#include "Box.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "World.hpp"

ValueStorage::ValueStorage() :
OptionRegistry_instance(OptionRegistry::getInstance())
{};

ValueStorage::~ValueStorage() {};


bool ValueStorage::isCurrentValuePresent(const char *name) const
{
  return (CurrentValueMap.find(name) != CurrentValueMap.end());
}

void ValueStorage::queryCurrentValue(const char * name, const atom * &_T)
{
  int atomID = -1;
  if (typeid( atom *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    atomID = lexical_cast<int>(CurrentValueMap[name].c_str());
    CurrentValueMap.erase(name);
  } else if (typeid( const atom *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    atomID = lexical_cast<int>(CurrentValueMap[name].c_str());
    CurrentValueMap.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getAtom(AtomById(atomID));
}

void ValueStorage::queryCurrentValue(const char * name, const element * &_T)  {
  int Z = -1;
  if (typeid(const element *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    Z = lexical_cast<int>(CurrentValueMap[name].c_str());
    CurrentValueMap.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getPeriode()->FindElement(Z);
}

void ValueStorage::queryCurrentValue(const char * name, const molecule * &_T) {
  int molID = -1;
  if (typeid( const molecule *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    molID = lexical_cast<int>(CurrentValueMap[name].c_str());
    CurrentValueMap.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
  _T = World::getInstance().getMolecule(MoleculeById(molID));
}

void ValueStorage::queryCurrentValue(const char * name, class Box &_T) {
  RealSpaceMatrix M;
  double tmp;
  if (typeid( Box ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValueMap[name]);
    stream >> tmp;
    M.set(0,0,tmp);
    stream >> tmp;
    M.set(0,1,tmp);
    M.set(1,0,tmp);
    stream >> tmp;
    M.set(0,2,tmp);
    M.set(2,0,tmp);
    stream >> tmp;
    M.set(1,1,tmp);
    stream >> tmp;
    M.set(1,2,tmp);
    M.set(2,1,tmp);
    stream >> tmp;
    M.set(2,2,tmp);
    _T = M;
    CurrentValueMap.erase(name);
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, class Vector &_T) {
  if (typeid( Vector ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    for (size_t i = 0; i < NDIM; ++i) {
      stream >> _T[i];
      ASSERT(!stream.fail(),
          "ValueStorage::queryCurrentValue() - Vector in value map has only "+toString(i)+" components!");
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, class BoxVector &_T) {
  if (typeid( BoxVector ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    for (size_t i = 0; i < NDIM; ++i) {
      stream >> _T[i];
      ASSERT(!stream.fail(),
          "ValueStorage::queryCurrentValue() - BoxVector in value map has only "+toString(i)+" components!");
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, std::vector<const atom *>&_T)
{
  int atomID = -1;
  size_t count = 0;
  atom *Walker = NULL;
  if (typeid( std::vector<const atom *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    while (!stream.fail()) {
      stream >> atomID >> ws;
      ASSERT((!stream.fail()) || (count != 0),
          "ValueStorage::queryCurrentValue() - Atom is missing id!");
      if (!stream.fail()) {
        Walker = World::getInstance().getAtom(AtomById(atomID));
        if (Walker != NULL)
          _T.push_back(Walker);
        atomID = -1;
        Walker = NULL;
        ++count;
      }
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, std::vector<const element *>&_T)
{
  int Z = -1;
  size_t count = 0;
  const element *elemental = NULL;
  if (typeid( std::vector<const element *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    while (!stream.fail()) {
      stream >> Z >> ws;
      ASSERT((!stream.fail()) || (count != 0),
          "ValueStorage::queryCurrentValue() - atomic number is missing!");
      if (!stream.fail()) {
        elemental = World::getInstance().getPeriode()->FindElement(Z);
        if (elemental != NULL)
          _T.push_back(elemental);
        Z = -1;
        ++count;
      }
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, std::vector<const molecule *>&_T)
{
  int molID = -1;
  size_t count = 0;
  molecule *mol = NULL;
  if (typeid( std::vector<const molecule *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    while (!stream.fail()) {
      stream >> molID >> ws;
      ASSERT((!stream.fail()) || (count != 0),
          "ValueStorage::queryCurrentValue() - molecule is missing id!");
      if (!stream.fail()) {
        mol = World::getInstance().getMolecule(MoleculeById(molID));
        if (mol != NULL)
          _T.push_back(mol);
        molID = -1;
        mol = NULL;
        ++count;
      }
    }
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::queryCurrentValue(const char * name, boost::filesystem::path&_T)
{
  std::string tmp;
  if (typeid( boost::filesystem::path ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    if (CurrentValueMap.find(name) == CurrentValueMap.end())
      throw MissingValueException(__FILE__, __LINE__);
    std::istringstream stream(CurrentValueMap[name]);
    CurrentValueMap.erase(name);
    stream >> tmp >> ws;
    ASSERT(!stream.fail(),
        "ValueStorage::queryCurrentValue() - boost::filesystem::path is missing!");
    _T = tmp;
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, const atom * &_T)
{
  if (typeid( const atom *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    stream << _T->getId();
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, const element * &_T)
{
  if (typeid( const element *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    stream << _T->getAtomicNumber();
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, const molecule * &_T)
{
  if (typeid( const molecule *) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    stream << _T->getId();
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, class Box &_T)
{
  const RealSpaceMatrix &M = _T.getM();
  if (typeid( Box ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    stream << M.at(0,0) << " ";
    stream << M.at(0,1) << " ";
    stream << M.at(0,2) << " ";
    stream << M.at(1,1) << " ";
    stream << M.at(1,2) << " ";
    stream << M.at(2,2) << " ";
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, class Vector &_T)
{
  if (typeid( Vector ) == *(OptionRegistry_instance.getOptionByName(name)->getType())){
    std::ostringstream stream;
    stream << _T[0] << " ";
    stream << _T[1] << " ";
    stream << _T[2] << " ";
    CurrentValueMap[name] = stream.str();
  } else  if (typeid( BoxVector ) == *(OptionRegistry_instance.getOptionByName(name)->getType())){
    std::ostringstream stream;
    stream << _T[0] << " ";
    stream << _T[1] << " ";
    stream << _T[2] << " ";
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, std::vector<const atom *>&_T)
{
  if (typeid( std::vector<const atom *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    for (std::vector<const atom *>::const_iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->getId() << " ";
    }
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, std::vector<const element *>&_T)
{
  if (typeid( std::vector<const element *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    for (std::vector<const element *>::const_iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->getAtomicNumber() << " ";
    }
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, std::vector<const molecule *>&_T)
{
  if (typeid( std::vector<const molecule *> ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    for (std::vector<const molecule *>::const_iterator iter = _T.begin(); iter != _T.end(); ++iter) {
      stream << (*iter)->getId() << " ";
    }
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

void ValueStorage::setCurrentValue(const char * name, boost::filesystem::path &_T)
{
  if (typeid( boost::filesystem::path ) == *(OptionRegistry_instance.getOptionByName(name)->getType())) {
    std::ostringstream stream;
    stream << _T.string();
    CurrentValueMap[name] = stream.str();
  } else
    throw IllegalTypeException(__FILE__,__LINE__);
}

const std::string ValueStorage::getCurrentValue(std::string actionname) {
  return CurrentValueMap[actionname];
}

CONSTRUCT_SINGLETON(ValueStorage)
