/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RemoveAction.cpp
 *
 *  Created on: Dez 8, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include <map>
#include <vector>

#include "Actions/UndoRedoHelpers.hpp"
#include "Atom/AtomicInfo.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include "RemoveAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "RemoveAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeRemoveAction::performCall() {

  std::vector<molecule *> molecules = World::getInstance().getSelectedMolecules();
  // check selected molecules
  if (molecules.empty()) {
    STATUS("No molecules selected.");
    return Action::failure;
  }

  STATUS("Removing "+toString(molecules.size())+" molecules.");
  // create undo state
  molecules_atoms_t MoleculeWalkers;
  for (std::vector<molecule *>::const_iterator iter = molecules.begin();
      iter != molecules.end(); ++iter) {
    std::vector<AtomicInfo> Walkers;
    for (molecule::const_iterator atomiter = const_cast<const molecule *>(*iter)->begin();
        atomiter != const_cast<const molecule *>(*iter)->end(); ++atomiter) {
      Walkers.push_back(**atomiter);
    }
    MoleculeWalkers.insert( std::make_pair( (*iter)->getId(), Walkers ) );
  }
  MoleculeRemoveState *UndoState = new MoleculeRemoveState(MoleculeWalkers, params);

  // remove all selected atoms
  for (std::vector<molecule *>::iterator iter = molecules.begin();
      iter != molecules.end(); ++iter)
    removeAtomsinMolecule(*iter);
  molecules.clear();

  return ActionState::ptr(UndoState);
}

ActionState::ptr MoleculeRemoveAction::performUndo(ActionState::ptr _state) {
  MoleculeRemoveState *state = assert_cast<MoleculeRemoveState*>(_state.get());

  // add all removed atoms again
  if (AddMoleculesFromAtomicInfo(state->MoleculeWalkers))
    return ActionState::ptr(_state);
  else {
    STATUS("Failed to re-add removed atoms in their molecules.");
    return Action::failure;
  }

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeRemoveAction::performRedo(ActionState::ptr _state){
  MoleculeRemoveState *state = assert_cast<MoleculeRemoveState*>(_state.get());

  // simply, remove molecules again, we have all the info
  for (molecules_atoms_t::const_iterator iter = state->MoleculeWalkers.begin();
      iter != state->MoleculeWalkers.end(); ++iter) {
    molecule * mol = World::getInstance().getMolecule(MoleculeById(iter->first));
    if (mol != NULL)
      removeAtomsinMolecule(mol);
  }

  return ActionState::ptr(_state);
}

bool MoleculeRemoveAction::canUndo() {
  return true;
}

bool MoleculeRemoveAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
