/*
 * Extractors.hpp
 *
 *  Created on: 15.10.2012
 *      Author: heber
 */

#ifndef TRAININGDATA_EXTRACTORS_HPP_
#define TRAININGDATA_EXTRACTORS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/bimap.hpp>
#include <boost/bimap/set_of.hpp>
#include <boost/bimap/multiset_of.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/breadth_first_search.hpp>
#include <boost/graph/subgraph.hpp>
#include <boost/function.hpp>

#include <map>
#include <set>

#include "Fragmentation/EdgesPerFragment.hpp"
#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

class BindingModel;
class Fragment;
class HomologyGraph;

/** Namespace containing all simple extractor functions.
 *
 * Extractor functions extract distances from a given fragment matching with
 * a given set of particle types (i.e. elements, e.h. H2O).
 * Filter functions extract a subset of distances from a given set of distances
 * to be used with a specific model.
 *
 * To this end, each FunctionModel has both a filter and an extractor function.
 *
 * The functions in this namespace act as helpers or basic building blocks in
 * constructing such filters and extractors.
 *
 */
namespace Extractors {
  typedef Fragment::charges_t::const_iterator chargeiter_t;
  typedef std::vector<chargeiter_t> chargeiters_t;

  typedef size_t count_t;
  typedef Fragment::atomicNumber_t element_t;
  typedef std::map< element_t, count_t> elementcounts_t;
  typedef std::map< element_t, chargeiters_t > elementtargets_t;
  typedef std::vector< chargeiters_t > targets_per_combination_t;
  //!> typedef for particle designation
  typedef unsigned int ParticleType_t;
  //!> typedef for a vector of particle designations
  typedef std::vector<ParticleType_t> ParticleTypes_t;

  typedef size_t level_t;
  typedef size_t node_t;
  typedef std::multimap< level_t, node_t > nodes_per_level_t;
  typedef std::set<node_t> nodes_t;
  typedef std::set<nodes_t> set_of_nodes_t;

  typedef boost::bimap<
      boost::bimaps::set_of< size_t >,
      boost::bimaps::multiset_of< Extractors::ParticleType_t >
  > type_index_lookup_t;

  typedef std::set<node_t> set_type;
  typedef std::set<set_type> powerset_type;

  typedef boost::adjacency_list < boost::vecS, boost::vecS, boost::undirectedS,
      boost::property<boost::vertex_name_t, atomId_t>,
      boost::property<boost::vertex_color_t, boost::default_color_type> /* needed for limited-depth DFS,
      otherwise the property_map gets full size of graph */
      > UndirectedGraph;
  typedef boost::subgraph< UndirectedGraph > UndirectedSubgraph;

  typedef boost::property_map < UndirectedGraph, boost::vertex_index_t >::type index_map_t;

  typedef std::map< node_t, std::pair<Extractors::ParticleType_t, size_t> > node_FragmentNode_map_t;

  typedef std::map< argument_t::indices_t, size_t> argument_placement_map_t;

  typedef std::map<size_t, size_t> argindex_to_nodeindex_t;

  /**
   * I have no idea why this is so complicated with BGL ...
   *
   * This is taken from the book "The Boost Graph Library: User Guide and Reference Manual, Portable Documents",
   * chapter "Basic Graph Algorithms", example on calculating the bacon number.
   */
  template <typename DistanceMap>
  class distance_recorder : public boost::default_bfs_visitor
  {
  public:
    distance_recorder(DistanceMap dist) : d(dist) {}

    template <typename Edge, typename Graph>
    void tree_edge(Edge e, const Graph &g) const {
      typename boost::graph_traits<Graph>::vertex_descriptor u = source(e,g), v = target(e,g);
      d[v] = d[u] + 1;
    }

  private:
    DistanceMap d;
  };

  template <typename DistanceMap>
  distance_recorder<DistanceMap> record_distance(DistanceMap d)
  {
    return distance_recorder<DistanceMap>(d);
  }

  HomologyGraph createHomologyGraphFromNodes(
      const nodes_t &nodes,
      const type_index_lookup_t &type_index_lookup,
      const UndirectedGraph &graph,
      const index_map_t &index_map
      );

  void generateAllInducedConnectedSubgraphs(
      const size_t N,
      const level_t level,
      const nodes_t &nodes,
      set_of_nodes_t &set_of_nodes,
      const nodes_per_level_t &nodes_per_level,
      const UndirectedGraph &graph,
      const std::vector<size_t> &_distance,
      const index_map_t &index_map);

  /** Namespace for some internal helper functions.
   *
   */
  namespace _detail {

    /** Counts all same elements in the vector and places into map of elements.
     *
     * \param elements vector of elements
     * \return count of same element in vector
     */
    elementcounts_t getElementCounts(
        const Fragment::atomicnumbers_t elements
        );

  }

  /** Gather all distances from a given set of positions.
   *
   *  Here, we only return one of the two equal distances.
   *
   * \param positions all nuclei positions
   * \param atomicNumber all nuclei atomic numbers
   * \param edges edges of the fragment's bond graph
   * \param globalid index to associated in argument_t with
   * \return vector of argument_ , each with a distance
   */
  FunctionModel::arguments_t
  gatherAllSymmetricDistanceArguments(
      const Fragment::positions_t& positions,
      const Fragment::atomicnumbers_t& atomicnumbers,
      const FragmentationEdges::edges_t &edges,
      const size_t globalid);

  /** Simple extractor of all unique pair distances of a given \a fragment, where
   * the first index is less than the second one.
   *
   * \param positions all nuclei positions
   * \param atomicNumber all nuclei atomic numbers
   * \param edges edges of the fragment's bond graph
   * \param index index refers to the index within the global set of configurations
   * \return vector of of argument_t containing all found distances
   */
  inline FunctionModel::arguments_t gatherAllSymmetricDistances(
      const Fragment::positions_t& positions,
      const Fragment::atomicnumbers_t& atomicnumbers,
      const FragmentationEdges::edges_t &edges,
      const size_t index
      ) {
    // get distance out of Fragment
    return gatherAllSymmetricDistanceArguments(positions, atomicnumbers, edges, index);
  }

  /** Filter the arguments to select only these required by the model.
   *
   * \warning this is meant as a faster way of getting the arguments for simple
   * pair potentials. In any other case, one should use filterArgumentsByBindingModel()
   *
   * \param listargs list of arguments to reorder each
   * \param _graph contains binding model of graph
   * \param _types particle type vector
   * \return reordered args
   */
  FunctionModel::list_of_arguments_t filterArgumentsByParticleTypes(
      const FunctionModel::arguments_t &args,
      const HomologyGraph &_graph,
      const ParticleTypes_t &_types,
      const BindingModel &_bindingmodel
      );

  /** Filter and reorder the arguments to bring adjacent ones together.
   *
   * We need to find all matching subgraphs (given by \a _bindingmodel) in the
   * given homology graph (given by \a _graph) of the fragment molecule.
   * This means filtering down to the desired particle types and then find
   * all possible matching subgraphs in each of argument lists, \a eachargs.
   *
   * \param listargs list of arguments to filter and order appropriately
   * \param _graph contains binding model of graph
   * \param _types particle type vector
   * \return reordered args
   */
  FunctionModel::list_of_arguments_t filterArgumentsByBindingModel(
      const FunctionModel::arguments_t &args,
      const HomologyGraph &_graph,
      const ParticleTypes_t &_types,
      const BindingModel &_bindingmodel
      );

  /** Combines two argument lists by sorting and making unique.
   *
   * @param firstargs first list of arguments
   * @param secondargs second list of arguments
   * @return concatenated lists
   */
  FunctionModel::arguments_t combineArguments(
      const FunctionModel::arguments_t &firstargs,
      const FunctionModel::arguments_t &secondargs);

  /** Combines two argument lists by concatenation.
   *
   * @param firstargs first list of arguments
   * @param secondargs second list of arguments
   * @return concatenated lists
   */
  FunctionModel::arguments_t concatenateArguments(
      const FunctionModel::arguments_t &firstargs,
      const FunctionModel::arguments_t &secondargs);

  /** Combines two argument lists by concatenation.
   *
   * @param firstlistargs first list of argument tuples
   * @param secondlistargs second list of argument tuples
   * @return concatenated lists
   */
  FunctionModel::list_of_arguments_t concatenateListOfArguments(
      const FunctionModel::list_of_arguments_t &firstlistargs,
      const FunctionModel::list_of_arguments_t &secondlistargs);

}; /* namespace Extractors */


#endif /* TRAININGDATA_EXTRACTORS_HPP_ */
