/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LinkedCell_ViewUnitTest.cpp
 *
 *  Created on: Nov 18, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <algorithm>
#include <list>

#include "Box.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/defs.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinkedCell/LinkedCell.hpp"
#include "LinkedCell/LinkedCell_Model.hpp"
#include "LinkedCell/LinkedCell_View.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "LinkedCell/unittests/defs.hpp"

#include "LinkedCell_ViewUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( LinkedCell_ViewTest );


void LinkedCell_ViewTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  //setVerbosity(3);

  // create diag(20.) matrix
  RealSpaceMatrix BoxM;
  BoxM.setIdentity();
  BoxM *= 20.;

  // create Box with this matrix
  domain = new Box(BoxM);

  // create LinkedCell structure with this Box
  LCImpl = new LinkedCell::LinkedCell_Model(EDGELENGTH, *domain);

  // create a list of nodes and add to LCImpl
  std::vector< Vector > VectorList;
  for (size_t i=0;i<((size_t)floor(NUMBERCELLS));++i)
    VectorList.push_back(Vector((double)i*EDGELENGTH,(double)i*EDGELENGTH,(double)i*EDGELENGTH));
  for (size_t i=0;i<VectorList.size();++i) {
    TesselPoint * Walker = new TesselPoint();
    Walker->setName(std::string("Walker")+toString(i));
    Walker->setPosition(VectorList[i]);
    NodeList.insert(Walker);
    LCImpl->addNode(Walker);
  }

  // create LinkedCell_View from that
  LC = new LinkedCell::LinkedCell_View(*LCImpl);
}


void LinkedCell_ViewTest::tearDown()
{
  delete LC;
  delete LCImpl;
  delete domain;

  // remove all nodes again
  for (PointSet::iterator iter = NodeList.begin();
      !NodeList.empty();
      iter = NodeList.begin()) {
    delete *iter;
    NodeList.erase(iter);
  }
}


/** UnitTest for getAllNeighbors()
 */
void LinkedCell_ViewTest::getAllNeighborsTest()
{
  // define some center vector
  Vector center(DOMAINLENGTH/2.,DOMAINLENGTH/2.,DOMAINLENGTH/2.);

  // get LinkedList from LC
  const double distance = 2.;
  LinkedCell::LinkedList NeighborList = LC->getAllNeighbors(distance, center);
//  for (LinkedCell::LinkedList::const_iterator iter = NeighborList.begin();
//      iter != NeighborList.end(); ++iter)
//    std::cout << **iter << " is in returned neighbor list." << std::endl;

  // gather points from NodeList
  LinkedCell::LinkedList ComparisonList;
  for (PointSet::const_iterator iter = NodeList.begin(); iter != NodeList.end(); ++iter)
    if (center.DistanceSquared((*iter)->getPosition()) <= distance*distance) {
      ComparisonList.push_back(*iter);
      //std::cout << **iter << " is inside of " << center << " plus " << distance << "." << std::endl;
    }
  // check that we get at least as many as needed
  CPPUNIT_ASSERT(ComparisonList.size() <= NeighborList.size());

  // sort lists to ease comparison
  ComparisonList.sort();
  NeighborList.sort();

  // check element-wise and skip unrequired ones
  LinkedCell::LinkedList::iterator iter1 = ComparisonList.begin();
  LinkedCell::LinkedList::iterator iter2 = NeighborList.begin();
  for(;(iter1 != ComparisonList.end()) && (iter2 != NeighborList.end()); ++iter1, ++iter2) {
    while (*iter1 != *iter2) {
      CPPUNIT_ASSERT( iter2 != NeighborList.end() );
      ++iter2;
    }
    //std::cout << **iter1 << " == " << **iter2 << std::endl;
    CPPUNIT_ASSERT( iter2 != NeighborList.end() );
  }
}

/** UnitTest for getPointsInsideSphere()
 */
void LinkedCell_ViewTest::getPointsInsideSphereTest()
{
  // define some center vector
  Vector center(DOMAINLENGTH/2.,DOMAINLENGTH/2.,DOMAINLENGTH/2.);

  // get LinkedList from LC
  const double distance = 3.;
  LinkedCell::LinkedList NeighborList = LC->getPointsInsideSphere(distance, center);
//  for (LinkedCell::LinkedList::const_iterator iter = NeighborList.begin();
//      iter != NeighborList.end(); ++iter)
//    std::cout << **iter << " is in returned restricted neighbor list." << std::endl;

  // gather points from NodeList
  LinkedCell::LinkedList ComparisonList;
  for (PointSet::const_iterator iter = NodeList.begin(); iter != NodeList.end(); ++iter)
    if (center.DistanceSquared((*iter)->getPosition()) <= distance*distance) {
      ComparisonList.push_back(*iter);
      //std::cout << **iter << " is inside of " << center << " plus " << distance << "." << std::endl;
    }
  // check that we get at least as many as needed
  CPPUNIT_ASSERT(ComparisonList.size() == NeighborList.size());

  // sort lists to ease comparison
  ComparisonList.sort();
  NeighborList.sort();

  // check element-wise and skip unrequired ones
  LinkedCell::LinkedList::iterator iter1 = ComparisonList.begin();
  LinkedCell::LinkedList::iterator iter2 = NeighborList.begin();
  for(;(iter1 != ComparisonList.end()) && (iter2 != NeighborList.end()); ++iter1, ++iter2) {
    //std::cout << **iter1 << " == " << **iter2 << std::endl;
    CPPUNIT_ASSERT( *iter1 == *iter2 );
  }
}


LinkedCell::LinkedCell_View returnView(LinkedCell::LinkedCell_View &view)
{
  return view;
}

LinkedCell::LinkedCell_View returnCopiedView(LinkedCell::LinkedCell_View &view)
{
  LinkedCell::LinkedCell_View retview(view);
  return retview;
}

/** UnitTest on whether counting in RAIIMap works
 */
void LinkedCell_ViewTest::RAIIMapTest()
{
  CPPUNIT_ASSERT_EQUAL( (size_t)1, LinkedCell::LinkedCell_View::RAIIMap.size() );

  // check that we are present
  LinkedCell::LinkedCell_View::ModelInstanceMap::iterator iter =
      LinkedCell::LinkedCell_View::RAIIMap.find(LC);
  CPPUNIT_ASSERT( iter != LinkedCell::LinkedCell_View::RAIIMap.end() );
  CPPUNIT_ASSERT( *iter == LC );

  // check that we are the only value present
  ++iter;
  CPPUNIT_ASSERT( iter == LinkedCell::LinkedCell_View::RAIIMap.end() );

  // add another view and check that there is not assertion
  LinkedCell::LinkedCell_View *view = NULL;
  CPPUNIT_ASSERT_NO_THROW( view = new LinkedCell::LinkedCell_View(*LCImpl) );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, LinkedCell::LinkedCell_View::RAIIMap.size() );
  delete view;
  CPPUNIT_ASSERT_EQUAL( (size_t)1, LinkedCell::LinkedCell_View::RAIIMap.size() );

  // copy current view
  {
    LinkedCell::LinkedCell_View view(*LC);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, LinkedCell::LinkedCell_View::RAIIMap.size() );
    LinkedCell::LinkedCell_View view2 = *LC;
    CPPUNIT_ASSERT_EQUAL( (size_t)3, LinkedCell::LinkedCell_View::RAIIMap.size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)1, LinkedCell::LinkedCell_View::RAIIMap.size() );

  // with function returning view
  {
    LinkedCell::LinkedCell_View view = returnView(*LC);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, LinkedCell::LinkedCell_View::RAIIMap.size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)1, LinkedCell::LinkedCell_View::RAIIMap.size() );

  // with function returning copied view
  {
    LinkedCell::LinkedCell_View view = returnCopiedView(*LC);
    CPPUNIT_ASSERT_EQUAL( (size_t)2, LinkedCell::LinkedCell_View::RAIIMap.size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)1, LinkedCell::LinkedCell_View::RAIIMap.size() );
}

