/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CommandLineParser.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <boost/filesystem.hpp>
#include <boost/program_options.hpp>
#include <fstream>
#include <iostream>
#include <map>

#include "Actions/Action.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionTraits.hpp"
#include "Actions/OptionRegistry.hpp"
#include "Actions/OptionTrait.hpp"
#include "Actions/Values.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "CommandLineParser.hpp"
#include "CommandLineParser_validate.hpp"

#include "Patterns/Singleton_impl.hpp"

class element;

/** Constructor of class CommandLineParser.
 *
 */
CommandLineParser::CommandLineParser() :
  analysis("Analysis options"),
  atom("Atom options"),
  command("Command options"),
  fragmentation("Fragmentation options"),
  molecule("Molecule options"),
  parser("Parser options"),
  selection("Selection options"),
  tesselation("Tesselation options"),
  world("World options"),
  options("Secondary options")
{
  // put all options lists into a lookup
  CmdParserLookup["analysis"] = &analysis;
  CmdParserLookup["atom"] = &atom;
  CmdParserLookup["command"] = &command;
  CmdParserLookup["edit"] = &edit;
  CmdParserLookup["fragmentation"] = &fragmentation;
  CmdParserLookup["molecule"] = &molecule;
  CmdParserLookup["parser"] = &parser;
  CmdParserLookup["selection"] = &selection;
  CmdParserLookup["tesselation"] = &tesselation;
  CmdParserLookup["world"] = &world;
  CmdParserLookup["options"] = &options;
}

/** Destructor of class CommandLineParser.
 *
 */
CommandLineParser::~CommandLineParser()
{}

/** Initializes command arguments to accept.
 * Goes through ActionRegistry and puts all actions therein into the map.
 */
void CommandLineParser::InitializeCommandArguments()
{
  ActionRegistry &AR = ActionRegistry::getInstance();
  bool ActionAlreadyAdded_flag = false;
  for (ActionRegistry::const_iterator actioniter = AR.getBeginIter(); actioniter != AR.getEndIter(); ++actioniter) {
    ActionAlreadyAdded_flag = false;
    Action* const currentAction = actioniter->second;
    //std::cout << "Current Action to initialize is: " << actioniter->first << std::endl;

    for (ActionTraits::options_const_iterator optioniter = currentAction->Traits.getBeginIter();
        optioniter != currentAction->Traits.getEndIter();
        ++optioniter) {
      if (optioniter->first == actioniter->first)
        ActionAlreadyAdded_flag = true;
      ASSERT( OptionRegistry::getInstance().isOptionPresentByName(optioniter->first),
          "CommandLineParser::Init() - Option not present in OptionRegistry." );
      const OptionTrait* const currentOption = OptionRegistry::getInstance().getOptionByName(optioniter->first);
      // add the option
      std::cout << "Registering Option "
          << currentOption->getName()
          << " with type '" << currentOption->getTypeName() << "' "
          << " with description '" << currentOption->getDescription() << "' ";
      if (currentOption->hasShortForm())
        std::cout << ", with short form " << currentOption->getShortForm();
      else
        std::cout << ", with no short form ";
      if (currentOption->hasDefaultValue())
        std::cout << ", with default value " << currentOption->getDefaultValue();
      else
        std::cout << ", with no default value ";
      std::cout << std::endl;

      AddOptionToParser(currentOption, (CmdParserLookup["options"]));
    }

    if (!ActionAlreadyAdded_flag) {
      // add the action
      std::cout << "Registering Action "
          << currentAction->Traits.getName()
          << " in menu " << currentAction->Traits.getMenuName()
          << " with type '" << currentAction->Traits.getTypeName() << "' "
          << " with description '" << currentAction->Traits.getDescription() << "' ";
      if (currentAction->Traits.hasShortForm())
        std::cout << ", with short form " << currentAction->Traits.getShortForm();
      else
        std::cout << ", with no short form ";
      if (currentAction->Traits.hasDefaultValue())
        std::cout << ", with default value " << currentAction->Traits.getDefaultValue();
      else
        std::cout << ", with no default value ";
      std::cout  << std::endl;

      ASSERT(CmdParserLookup.find(currentAction->Traits.getMenuName()) != CmdParserLookup.end(),
          "CommandLineParser: boost::program_options::options_description for this Action not present.");
      AddOptionToParser(dynamic_cast<const OptionTrait * const>(&(currentAction->Traits)),(CmdParserLookup[currentAction->Traits.getMenuName()]));
    }
  }
  // note: positioning is not important on the command line
}

/** Adds an Action or Option to the CommandLineParser.
 * Note that Action is derived from Option(Trait)
 *
 * This ugly switch function is necessary because of the compile-time problem:
 * po::value<T> has to be instantiated at compile-time however we do know the type not until run-time.
 * Not even a templated function like po::value<T> getProgramOptionValuefromType() does help, specialized
 * to each available type, as the signatures of all the functions differ. Hence, they cannot not put into
 * one type_info -> po::value<T> map ...
 *
 * \param *currentOption pointer to Action/Option to add
 * \param *OptionList program_options list to add to
 */
void CommandLineParser::AddOptionToParser(const OptionTrait * const currentOption, po::options_description* OptionList)
{
  // check whether dynamic_cast in Init() suceeded
  ASSERT(currentOption != NULL, "CommandLineParser::AddOptionToParser() - currentOption is NULL!");
  // add other options
  std::cout << "Adding Action " << currentOption->getName() << " with type " << currentOption->getType()->name() << " and KeyandShortform " << currentOption->getKeyAndShortForm() << " to CommandLineParser." << std::endl;
  switch(TypeToEnums.getEnumforType(currentOption->getType())) {
    default:
    case TypeEnumContainer::NoneType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(), currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::BooleanType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  po::value < bool >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str())) :
                  po::value < bool >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::BoxType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < BoxValue >()->default_value(boost::lexical_cast<BoxValue>(currentOption->getDefaultValue().c_str())) :
                  po::value < BoxValue >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::FileType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < boost::filesystem::path >()->default_value(boost::lexical_cast<boost::filesystem::path>(currentOption->getDefaultValue().c_str())) :
                  po::value < boost::filesystem::path >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfFilesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<boost::filesystem::path> >()->default_value(boost::lexical_cast< std::vector<boost::filesystem::path> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<boost::filesystem::path> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::IntegerType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  po::value < int >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str())) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfIntegersType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<int> >()->default_value(boost::lexical_cast< std::vector<int> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::DoubleType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  po::value < double >()->default_value(boost::lexical_cast<double>(currentOption->getDefaultValue().c_str())) :
                  po::value < double >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfDoublesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<double> >()->default_value(boost::lexical_cast< std::vector<double> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<double> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::StringType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  po::value < std::string >()->default_value(currentOption->getDefaultValue()) :
                  po::value < std::string >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfStringsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<std::string> >()->default_value(boost::lexical_cast< std::vector<std::string> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<std::string> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::VectorType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < VectorValue >()->default_value(boost::lexical_cast<VectorValue>(currentOption->getDefaultValue().c_str())) :
                  po::value < VectorValue >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfVectorsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<VectorValue> >()->default_value(boost::lexical_cast< std::vector<VectorValue> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<VectorValue> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::MoleculeType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < const molecule * >()->default_value(boost::lexical_cast<const molecule *>(currentOption->getDefaultValue().c_str())) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfMoleculesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const molecule *> >()->default_value(boost::lexical_cast< std::vector<const molecule *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::AtomType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  po::value < int >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str())) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfAtomsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const atom *> >()->default_value(boost::lexical_cast< std::vector<const atom *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ElementType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < const element * >()->default_value(boost::lexical_cast<const element *>(currentOption->getDefaultValue().c_str())) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfElementsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const element *> >()->default_value(boost::lexical_cast< std::vector<const element *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
  }
}

/** States whether there are command line arguments.
 * \return true - there are none, false - there is at least one command line argument
 */
bool CommandLineParser::isEmpty()
{
  return vm.empty();
}

/** Sets the options.
 * \param _argc arg count from main()
 * \param **_argv argument array from main()
 */
void CommandLineParser::setOptions(int _argc, char **_argv)
{
  argc = _argc;
  argv = _argv;
  cmdline_options.add(analysis).add(atom).add(command).add(edit).add(fragmentation).add(molecule).add(parser).add(selection).add(tesselation).add(world).add(options);
  config_file_options.add(options);
  visible.add(analysis).add(atom).add(command).add(edit).add(fragmentation).add(molecule).add(parser).add(selection).add(tesselation).add(world).add(options);
}

/** Parses the command line arguments.
 * Calls program_options::store() and program_options::notify()
 */
void CommandLineParser::Parse()
{
  po::store(po::command_line_parser(argc,argv).options(cmdline_options).run(), vm);
  std::ifstream input;
  input.open("example.cfg");
  if (!input.fail())
    po::store(po::parse_config_file(input, config_file_options), vm);
  input.close();
  po::notify(vm);
}

/** Scan the argument list for -a or --arguments and store their order for later use.
 */
void CommandLineParser::scanforSequenceOfArguments()
{
  std::map <std::string, std::string> ShortFormToActionMap = getShortFormToActionMap();
  // go through all arguments
  for (int i=1;i<argc;i++) {
    (std::cout << Verbose(1) << "Checking on " << argv[i] << std::endl);
    // check whether they
    if (argv[i][0] == '-') { // .. begin with -
      (cout << Verbose(1) << "Possible argument: " << argv[i] << endl);
      if (argv[i][1] == '-') { // .. or --
        (cout << Verbose(1) << "Putting " << argv[i] << " into the sequence." << endl);
        SequenceOfActions.push_back(&(argv[i][2]));
        //  .. and check that next letter is not numeric, if so insert
      } else if (((argv[i][1] < '0') || (argv[i][1] > '9')) && ((argv[i][1] != '.'))) {
        std::map <std::string, std::string>::iterator iter = ShortFormToActionMap.find(&(argv[i][1]));
        if (iter != ShortFormToActionMap.end()) {
          (cout << Verbose(1) << "Putting " << iter->second << " for " << iter->first << " into the sequence." << endl);
          SequenceOfActions.push_back(iter->second);
        }
      }
    }
  }
}

/** Makes the Parser parse the command line options with current known options.
 * \param _argc arg count from main()
 * \param **_argv argument array from main()
 */
void CommandLineParser::Run(int _argc, char **_argv)
{
  setOptions(_argc,_argv);
  Parse();
  scanforSequenceOfArguments();
}

/** Go through all Actions and create a map from short form to their token.
 * \return map from Action's ShortForm to token.
 */
std::map <std::string, std::string> CommandLineParser::getShortFormToActionMap()
{
  std::map <std::string, std::string> result;

  ActionRegistry &AR = ActionRegistry::getInstance();
  for (ActionRegistry::const_iterator iter = AR.getBeginIter(); iter != AR.getEndIter(); ++iter)
    if ((iter->second)->Traits.hasShortForm()) {
      ASSERT(result.find((iter->second)->Traits.getShortForm()) == result.end(),
          "Short form "+toString((iter->second)->Traits.getShortForm())+
          " for action "+toString(iter->first)+" already present from "+
          std::string(result[(iter->second)->Traits.getShortForm()])+"!");
      result[(iter->second)->Traits.getShortForm()] = (iter->second)->getName();
    }

  return result;
}

CONSTRUCT_SINGLETON(CommandLineParser)
