/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * HomologyGraph_getFromKeyset.cpp
 *
 *  Created on: Sep 25, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Fragmentation/Homology/HomologyGraph.hpp"

#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Fragmentation/KeySet.hpp"
#include "Fragmentation/Summation/IndexSet.hpp"
#include "World.hpp"

// we have placed these functions into an extra module to allow compiling the
// unit tests which do no require them against dummy units which do not pull
// in all the cludder of World, atom, molecule, and so on ...

template <typename S, typename U>
void addNodeToMap(S &_map,  std::pair< U, size_t> _pair) {
  std::pair<typename S::iterator,bool> inserter =
      _map.insert( _pair );
  if (!inserter.second)
    inserter.first->second += _pair.second;
}

template <typename T>
const HomologyGraph::nodes_t getNodesFromSet(const std::set<T> &keyset)
{
  HomologyGraph::nodes_t nodes;
  for (typename std::set<T>::const_iterator iter = keyset.begin();
      iter != keyset.end(); ++iter) {
//    LOG(2, "DEBUG: Current global id is " << *iter << ".");
    const atom * const Walker = const_cast<const World &>(World::getInstance()).
        getAtom(AtomById(*iter));
    if (Walker != NULL) {
      size_t NoBonds = 0;
      const BondList& ListOfBonds = Walker->getListOfBonds();
      for (BondList::const_iterator bonditer = ListOfBonds.begin();
          bonditer != ListOfBonds.end(); ++bonditer) {
        const atom * const OtherWalker = (*bonditer)->GetOtherAtom(Walker);
        if (keyset.count(OtherWalker->getId()))
          ++NoBonds;
        else {
          // add as many bonds as saturation hydrogens
          NoBonds += (*bonditer)->getDegree();
          // add a saturation node for each, too
//          LOG(1, "DEBUG: Adding saturation node " << *Walker << ".");
          addNodeToMap(nodes, std::make_pair(FragmentNode(1, 1), (size_t)1) );
        }
      }
//      LOG(1, "DEBUG: Adding node " << *Walker << " with element "
//          << Walker->getElementNo() << " and " << NoBonds << " bonds.");
      addNodeToMap(nodes, std::make_pair(FragmentNode(Walker->getElementNo(), NoBonds), (size_t)1) );
    } else {
      ELOG(3, "Skipping id " << *iter << ", is not associated with any atom.");
    }
  }
  return nodes;
}

template <typename T>
const HomologyGraph::edges_t getEdgesFromSet(const std::set<T> &keyset)
{
  HomologyGraph::edges_t edges;
  for (typename std::set<T>::const_iterator iter = keyset.begin();
      iter != keyset.end(); ++iter) {
//    LOG(2, "DEBUG: Current global id is " << *iter << ".");
    const atom * const Walker = const_cast<const World &>(World::getInstance()).
        getAtom(AtomById(*iter));
    if (Walker != NULL) {
      const BondList& ListOfBonds = Walker->getListOfBonds();
      for (BondList::const_iterator bonditer = ListOfBonds.begin();
          bonditer != ListOfBonds.end(); ++bonditer) {
        const atom * const OtherWalker = (*bonditer)->GetOtherAtom(Walker);
//        LOG(1, "DEBUG: Neighbor is " << OtherWalker->getId() << ".");
        if (keyset.count(OtherWalker->getId())) {
          if (Walker->getId() < OtherWalker->getId()) {
//            LOG(1, "DEBUG: Adding edge " << Walker->getId() << " and " << OtherWalker->getId() << ".");
            addNodeToMap(edges, std::make_pair(FragmentEdge( Walker->getElementNo(), OtherWalker->getElementNo()), (size_t)1) );
          }
        } else { // also add edge for each saturation hydrogen
          const size_t bonddegree = (*bonditer)->getDegree();
//          LOG(1, "DEBUG: Adding " << bonddegree << " saturation edge(s) " << Walker->getId() << " and " << OtherWalker->getId() << ".");
          addNodeToMap(edges, std::make_pair(FragmentEdge( Walker->getElementNo(), 1), bonddegree) );
        }
      }
    } else {
      ELOG(3, "Skipping id " << *iter << ", is not associated with any atom.");
    }
  }
  return edges;
}

namespace detail {
  const HomologyGraph::nodes_t getNodesFromKeySet(const KeySet &keyset) {
    return getNodesFromSet<int>(keyset);
  }
  const HomologyGraph::nodes_t getNodesFromIndexSet(const IndexSet &keyset) {
    return getNodesFromSet<size_t>(keyset);
  }
  const HomologyGraph::edges_t getEdgesFromKeySet(const KeySet &keyset) {
    return getEdgesFromSet<int>(keyset);
  }
  const HomologyGraph::edges_t getEdgesFromIndexSet(const IndexSet &keyset) {
    return getEdgesFromSet<size_t>(keyset);
  }
};  /* namespace detail */

