/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AdjacencyMatcher.cpp
 *
 *  Created on: Jul 19, 2015
 *      Author: heber
 *
 *
 *  This is mostly copy&pasted from AdjacencyMatcher.cpp
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include <boost/exception/all.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/tokenizer.hpp>

#include <algorithm>
#include <fstream>
#include <iostream>
#include <iterator>
#include <map>
#include <set>
#include <sstream>
#include <string>

#include "Adjacency.hpp"

#include "CodePatterns/toString.hpp"

typedef std::set<detail::Adjacency> adjacencies_t;

using namespace detail;

namespace bf = boost::filesystem;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_LineNumber,
		int> IllegalTesselationFileContent_LineNumber;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_Entries,
		int> IllegalTesselationFileContent_Entries;

typedef boost::error_info<
		struct tag_IllegalTesselationFileContent_Line,
		int> IllegalTesselationFileContent_Line;

//!> Exception for illegal lines encountered in adjacencies files
struct IllegalTesselationFileContentException:
		virtual boost::exception, virtual std::exception { };

Adjacency::index_t toIndex(const std::string &_element)
{
	Adjacency::index_t index;
	std::stringstream tokenstring(_element);
	tokenstring >> index;
	return index;
}

template <class T>
std::ostream& operator<<(
		std::ostream &_ost,
		const std::set<T> &elements)
{
	for( const T elem : elements)
		_ost << elem << std::endl;
	return _ost;
}

/** Parses a adjacencies file into a vector of Adjacencys.
 *
 * \param _filename name of file to parse
 * \return adjacencies_t structure filled with adjacency index triples
 */
const adjacencies_t parseFile(const boost::filesystem::path &_filename)
{
	adjacencies_t adjacencies;

	std::ifstream infile(_filename.string().c_str());
	boost::char_separator<char> sep(" \t");

	typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
	std::string line;
	while (!infile.eof()) {
		std::getline(infile, line);
		tokenizer tokens(line, sep);
		Adjacency::indices_t indices;
		const int entries = std::distance( tokens.begin(), tokens.end() );
		// check for empty lines
		if (entries == 0)
			continue;
		std::transform(
				tokens.begin(),
				tokens.end(),
				std::back_inserter(indices),
				toIndex);
		adjacencies.insert(Adjacency(indices));
	}
	infile.close();

	return adjacencies;
}

int main(int argc, char **argv) {

	/// check arguments and give usage
	if (argc <= 2) {
		std::cerr << "Usage: " << argv[0] << " <list of files>\n";
		std::cerr << "At least two adjacencies files need to be specified that contain\n";
		std::cerr << "each three indices per line.\n";

		return 255;
	}

	/// read in files and parse into structures
	typedef std::multimap<adjacencies_t, std::string> adjacencies_filename_t;
	//!> contains all adjacencies parsed from files
	adjacencies_filename_t adjacencyfiles;
	for (int i=1; i<argc;++i) {
		const bf::path filename(argv[i]);
		if (bf::exists(filename)) {
			try {
				adjacencies_t adjacencies = parseFile(filename);
				adjacencyfiles.insert(
						std::make_pair(adjacencies, filename.string()) );
			} catch (IllegalTesselationFileContentException &e) {
				std::cerr << "Illegal line at "
						<< *boost::get_error_info<IllegalTesselationFileContent_LineNumber>(e)
						<< " encountered with "
						<< *boost::get_error_info<IllegalTesselationFileContent_Entries>(e)
						<< " entries in "
						<< filename.string()
						<< ", aborting.\n";
				return 5;
			}
		} else {
			std::cerr << "The following file does not exist "
					<< filename.string() << ", aborting.\n";
			return 5;
		}
	}

	/// make comparison for equivalence
	assert( adjacencyfiles.size() >= 2 );
	adjacencies_filename_t::const_iterator advancer = adjacencyfiles.begin();
	adjacencies_filename_t::const_iterator iter = advancer++;

	bool AllEqual = true;
	while( advancer != adjacencyfiles.end() ) {
//	  std::cout << "Comparing " << *advancer << " with " << *iter << std::endl;
		if (advancer->first == iter->first) {
			std::cout << "MATCH: "
					<< advancer->second << " and " << iter->second
					<< std::endl;
		} else {
			std::cout << advancer->first << " and " << iter->first
					<< " don't match.\n";
			AllEqual = false;
		}
		iter = advancer++;
	}

	// exit (convert bool to int)
	return (AllEqual ? 0 : 1);
}
