/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MoleculeByIdAction.cpp
 *
 *  Created on: May 12, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "molecule.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "MoleculeByIdAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "MoleculeByIdAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr SelectionMoleculeByIdAction::performCall() {

  enum Sucess {
    NoStatus,
    AllMoleculesUnselected,
    MoleculesSelected,
    MoleculeMissing
  } status = NoStatus;

  const molids_t molids = params.molids.get();
  molids_t undomolids;
  undomolids.reserve(molids.size());
  for (molids_t::const_iterator iter = molids.begin(); iter != molids.end(); ++iter) {
    const molecule *Walker = const_cast<const World &>(World::getInstance()).
        getMolecule(MoleculeById(*iter));
    if (Walker != NULL) {
      if (!World::getInstance().isSelected(Walker)) {
        LOG(1, "Selecting mol " << Walker->getName());
        World::getInstance().selectMolecule(Walker);
        undomolids.push_back(*iter);
        if (status < MoleculeMissing)
          status = MoleculesSelected;
      } else {
        if (status == NoStatus)
          status = AllMoleculesUnselected;
      }
    } else {
      status = MoleculeMissing;
    }
  }
  LOG(0, World::getInstance().countSelectedMolecules() << " mols selected.");

  switch (status) {
    case MoleculeMissing:
      STATUS("Cannot find all mols with given ids.");
      return Action::failure;
      break;
    case AllMoleculesUnselected:
    case MoleculesSelected:
      return ActionState::ptr(new SelectionMoleculeByIdState(undomolids, params));
      break;
    default:
      STATUS("No mols have been selected.");
      return Action::failure;
      break;
  }
  return Action::failure;
}

ActionState::ptr SelectionMoleculeByIdAction::performUndo(ActionState::ptr _state) {
  SelectionMoleculeByIdState *state = assert_cast<SelectionMoleculeByIdState*>(_state.get());

  for (molids_t::const_iterator iter = state->undomolids.begin();
      iter != state->undomolids.end(); ++iter) {
    const molecule *Walker = const_cast<const World &>(World::getInstance()).
        getMolecule(MoleculeById(*iter));
    World::getInstance().unselectMolecule(Walker);
  }

  return ActionState::ptr(_state);
}

ActionState::ptr SelectionMoleculeByIdAction::performRedo(ActionState::ptr _state){
  SelectionMoleculeByIdState *state = assert_cast<SelectionMoleculeByIdState*>(_state.get());

  for (molids_t::const_iterator iter = state->undomolids.begin();
      iter != state->undomolids.end(); ++iter) {
    const molecule *Walker = const_cast<const World &>(World::getInstance()).
        getMolecule(MoleculeById(*iter));
    World::getInstance().selectMolecule(Walker);
  }

  return ActionState::ptr(_state);
}

bool SelectionMoleculeByIdAction::canUndo() {
  return true;
}

bool SelectionMoleculeByIdAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
