/*
 * ContinuousValue_impl.hpp
 *
 *  Created on: Sep 29, 2011
 *      Author: heber
 */

#ifndef CONTINUOUSVALUE_IMPL_HPP_
#define CONTINUOUSVALUE_IMPL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include <boost/any.hpp>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "Range.hpp"

#include "ContinuousValue.hpp"

/** Constructor of class DiscreteValue.
 */
template <class T>
ContinuousValue<T>::ContinuousValue() :
  ValueSet(false),
  ValidRangeSet(range<bool>(false, false))
{}

/** Constructor of class DiscreteValue with set of valid values.
 *
 * @param _ValidValues vector with all valid values
 */
template <class T>
ContinuousValue<T>::ContinuousValue(const range<T> &_ValidRange) :
  ValueSet(false),
  ValidRangeSet(range<bool>(true, true)),
  ValidRange(_ValidRange)
{
//  LOG(0, "STATUS: Valid range is now " << ValidRange << ".");
}

/** Destructor of class DiscreteValue.
 */
template <class T>
ContinuousValue<T>::~ContinuousValue()
{}

/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
bool ContinuousValue<T>::isValid(const T & _value) const
{
  return isValidValue(_value);
}

/** Compares this continuous value against another \a _instance.
 *
 * @param _instance other value to compare to
 * @return true - if value and valid ranges are the same, false - else
 */
template <class T>
bool ContinuousValue<T>::operator==(const ContinuousValue<T> &_instance) const
{
  bool status = true;
  status = status && (ValidRange == _instance.ValidRange);
  status = status && (ValueSet == _instance.ValueSet);
  if (ValueSet && _instance.ValueSet)
    status = status && (value == _instance.value);
  return status;
}

/** Getter of value, returning string.
 *
 * @return string value
 */
template <class T>
const T & ContinuousValue<T>::get() const
{
  ASSERT(ValueSet,
      "ContinuousValue<T>::get() - requesting unset value.");
  return value;
}

/** Setter of value for string
 *
 * @param _value string containing new value
 */
template <class T>
void ContinuousValue<T>::set(const T & _value)
{
  setValue(_value);
//  LOG(0, "STATUS: Value is now set to " << value << ".");
}

/** Setter for the valid range.
 *
 * If value is invalid in new range, we throw AssertFailure and set ValueSet to false.
 *
 * @param _range range (pair of values)
 */
template <class T>
void ContinuousValue<T>::setValidRange(const range<T> &_range)
{

  ValidRangeSet = range<bool>(true, true);
  ValidRange = _range;
  if (ValueSet) {
    //std::cout << "Checking whether " << value << " is in range " << ValidRange << "." << std::endl;
    if (!((ValidRange.isInRange(value)) || (value == ValidRange.last))) {
      //std::cout << "ValueSet to false." << std::endl;
      ValueSet = false;
      // have full check again in assert such that it appears in output, too
      ASSERT(ValidRange.isInRange(value) || (value == ValidRange.last),
          "ContinuousValue<T>::setValidRange() - new range "
          +toString(_range)+" invalidates current value "+toString(value)+".");
    }
  }
//  LOG(0, "STATUS: Valid range is now " << ValidRange << ".");
}

/** Getter for the valid range.
 *
 * @return _range range (pair of values)
 */
template <class T>
const range<T> & ContinuousValue<T>::getValidRange() const
{
  ASSERT(ValidRangeSet.first && ValidRangeSet.last,
      "ContinuousValue<T>::getValidRange() called though no valid range set so far.");
  return ValidRange;
}

/** Checks whether \a _value is a valid value.
 * \param _value value to check for validity.
 * \return true - \a _value is valid, false - is not
 */
template <class T>
bool ContinuousValue<T>::isValidValue(const T &_value) const
{
  bool isBefore = true;
  bool isBeyond = true;
  // check left boundary
  isBefore = !((!ValidRangeSet.first) || (!ValidRange.isBefore(_value)));
//  if (isBefore)
//    LOG(0, "INFO: " << _value << " is before " << ValidRange.first << ".");
  // check right boundary
  isBeyond = !((!ValidRangeSet.last) || (!ValidRange.isBeyond(_value)) || (_value == ValidRange.last));
//  if (isBeyond)
//    LOG(0, "INFO: " << _value << " is beyond " << ValidRange.last << ".");
  return (!isBefore) && (!isBeyond);
}


/** Sets the value.
 *
 * We check for its validity, otherwise we throw an Assert::AssertionFailure.
 *
 * @param _value const reference of value to set
 */
template <class T>
void ContinuousValue<T>::setValue(const T &_value)
{
  ASSERT(isValidValue(_value),
      "ContinuousValue<T>::setValue() - trying to set invalid value "+toString(_value)+".");
  if (!ValueSet)
    ValueSet = true;
  value = _value;
}

/** Getter for the set value.
 *
 * We check whether it has been set, otherwise we throw an Assert::AssertionFailure.
 *
 * @return set value
 */
template <class T>
const T &ContinuousValue<T>::getValue() const
{
  ASSERT(ValueSet,
      "ContinuousValue<T>::get() - value has never been set.");
  return value;
}

#endif /* CONTINUOUSVALUE_IMPL_HPP_ */
