/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomsWithinDistanceOfDescriptor.cpp
 *
 *  Created on: Dec 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "AtomsWithinDistanceOfDescriptor.hpp"
#include "AtomsWithinDistanceOfDescriptor_impl.hpp"

#include "Atom/atom.hpp"
#include "LinkedCell/LinkedCell_View.hpp"
#include "CodePatterns/Observer/ObservedContainer_impl.hpp"

using namespace std;


AtomsWithinDistanceOfDescriptor_impl::AtomsWithinDistanceOfDescriptor_impl(const double _distance, const Vector &_position) :
    distance(_distance),
    distanceSquared(distance*distance),
    position(_position)
{}

AtomsWithinDistanceOfDescriptor_impl::~AtomsWithinDistanceOfDescriptor_impl()
{}

bool AtomsWithinDistanceOfDescriptor_impl::predicate(std::pair<atomId_t,const atom*> atom) const
{
  return ((atom.second->getPosition().DistanceSquared(position) - distanceSquared) <= 0);
}

atom* AtomsWithinDistanceOfDescriptor_impl::find(){
  LinkedCell::LinkedCell_View view = World::getInstance().getLinkedCell(distance);
  LinkedCell::LinkedList list = view.getPointsInsideSphere(distance, position);
  return (list.begin()!=list.end())? static_cast<atom *>(const_cast<TesselPoint *>(*list.begin())):0;
}

const atom* AtomsWithinDistanceOfDescriptor_impl::find() const{
  LinkedCell::LinkedCell_View view = World::getInstance().getLinkedCell(distance);
  LinkedCell::LinkedList list = view.getPointsInsideSphere(distance, position);
  return (list.begin()!=list.end())? static_cast<const atom *>(*list.begin()):0;
}

std::vector<atom*> AtomsWithinDistanceOfDescriptor_impl::findAll(){
  LinkedCell::LinkedCell_View view = World::getInstance().getLinkedCell(distance);
  LinkedCell::LinkedList list = view.getPointsInsideSphere(distance, position);
  std::vector<atom*> res;
  for (LinkedCell::LinkedList::iterator iter = list.begin(); iter != list.end(); ++iter)
    res.push_back(static_cast<atom *>(const_cast<TesselPoint *>(*iter)));
  return res;
}

std::vector<const atom*> AtomsWithinDistanceOfDescriptor_impl::findAll() const {
  LinkedCell::LinkedCell_View view = World::getInstance().getLinkedCell(distance);
  LinkedCell::LinkedList list = view.getPointsInsideSphere(distance, position);
  std::vector<const atom*> res;
  for (LinkedCell::LinkedList::iterator iter = list.begin(); iter != list.end(); ++iter)
    res.push_back(static_cast<const atom *>(*iter));
  return res;
}

AtomDescriptor AtomsWithinDistanceOf(const double distance, const Vector &position)
{
  return AtomDescriptor(AtomDescriptor::impl_ptr(new AtomsWithinDistanceOfDescriptor_impl(distance, position)));
}
