/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * KeySetsContainer.cpp
 *
 *  Created on: Sep 15, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <fstream>
#include <sstream>

#include "CodePatterns/Log.hpp"

#include "Fragmentation/defs.hpp"
#include "Fragmentation/helpers.hpp"
#include "Helpers/defs.hpp"
#include "KeySetsContainer.hpp"

/** Constructor of KeySetsContainer class.
 */
KeySetsContainer::KeySetsContainer() :
  FragmentCounter(0),
  Order(0),
  FragmentsPerOrder(0)
{};

/** Destructor of KeySetsContainer class.
 */
KeySetsContainer::~KeySetsContainer() {
//  for(int i=FragmentCounter;i--;)
//    delete[](KeySets[i]);
//  for(int i=Order;i--;)
//    delete[](OrderSet[i]);
//  delete[](KeySets);
//  delete[](OrderSet);
//  delete[](AtomCounter);
//  delete[](FragmentsPerOrder);
};

/** Parsing KeySets into array.
 * \param *name directory with keyset file
 * \param *ACounter number of atoms per fragment
 * \param FCounter number of fragments
 * \return parsing succesful
 */
bool KeySetsContainer::ParseKeySets(const std::string name, const IntVector ACounter, const int FCounter) {
  ifstream input;
  char *FragmentNumber = NULL;
  stringstream file;
  char filename[1023];

  FragmentCounter = FCounter;
  DoLog(0) && (Log() << Verbose(0) << "Parsing key sets." << endl);
  KeySets.resize(FragmentCounter);
  file << name << FRAGMENTPREFIX << KEYSETFILE;
  input.open(file.str().c_str(), ios::in);
  if (input.fail()) {
    DoLog(0) && (Log() << Verbose(0) << endl << "KeySetsContainer::ParseKeySets: Unable to open " << file.str() << ", is the directory correct?" << endl);
    return false;
  }

  AtomCounter.resize(FragmentCounter);
  for(int i=0;(i<FragmentCounter) && (!input.eof());i++) {
    stringstream line;
    AtomCounter[i] = ACounter[i];
    // parse the values
    KeySets[i].resize(AtomCounter[i]);
    for(int j=AtomCounter[i];j--;)
      KeySets[i][j] = -1;
    FragmentNumber = FixedDigitNumber(FragmentCounter, i);
    //Log() << Verbose(0) << FRAGMENTPREFIX << FragmentNumber << "[" << AtomCounter[i] << "]:";
    delete[](FragmentNumber);
    input.getline(filename, 1023);
    line.str(filename);
    for(int j=0;(j<AtomCounter[i]) && (!line.eof());j++) {
      line >> KeySets[i][j];
      //Log() << Verbose(0) << " " << KeySets[i][j];
    }
    //Log() << Verbose(0) << endl;
  }
  input.close();
  return true;
};

/** Parse many body terms, associating each fragment to a certain bond order.
 * \return parsing succesful
 */
bool KeySetsContainer::ParseManyBodyTerms()
{
  int Counter;

  DoLog(0) && (Log() << Verbose(0) << "Creating Fragment terms." << endl);
  // scan through all to determine maximum order
  Order=0;
  for(int i=FragmentCounter;i--;) {
    Counter=0;
    for(int j=AtomCounter[i];j--;)
      if (KeySets[i][j] != -1)
        Counter++;
    if (Counter > Order)
      Order = Counter;
  }
  DoLog(0) && (Log() << Verbose(0) << "Found Order is " << Order << "." << endl);

  // scan through all to determine fragments per order
  FragmentsPerOrder.resize(Order);
//  for(int i=Order;i--;)
//    FragmentsPerOrder[i] = 0;
  for(int i=FragmentCounter;i--;) {
    Counter=0;
    for(int j=AtomCounter[i];j--;)
      if (KeySets[i][j] != -1)
        Counter++;
    FragmentsPerOrder[Counter-1]++;
  }
  for(int i=0;i<Order;i++)
    DoLog(0) && (Log() << Verbose(0) << "Found No. of Fragments of Order " << i+1 << " is " << FragmentsPerOrder[i] << "." << endl);

  // scan through all to gather indices to each order set
  OrderSet.resize(Order);
  for(int i=Order;i--;)
    OrderSet[i].resize(FragmentsPerOrder[i]);
  for(int i=Order;i--;)
    FragmentsPerOrder[i] = 0;
  for(int i=FragmentCounter;i--;) {
    Counter=0;
    for(int j=AtomCounter[i];j--;)
      if (KeySets[i][j] != -1)
        Counter++;
    OrderSet[Counter-1][FragmentsPerOrder[Counter-1]] = i;
    FragmentsPerOrder[Counter-1]++;
  }
  DoLog(0) && (Log() << Verbose(0) << "Printing OrderSet." << endl);
  for(int i=0;i<Order;i++) {
    for (int j=0;j<FragmentsPerOrder[i];j++) {
      DoLog(0) && (Log() << Verbose(0) << " " << OrderSet[i][j]);
    }
    DoLog(0) && (Log() << Verbose(0) << endl);
  }
  DoLog(0) && (Log() << Verbose(0) << endl);


  return true;
};

/** Compares each entry in \a *SmallerSet if it is containted in \a *GreaterSet.
 * \param GreaterSet index to greater set
 * \param SmallerSet index to smaller set
 * \return true if all keys of SmallerSet contained in GreaterSet
 */
bool KeySetsContainer::Contains(const int GreaterSet, const int SmallerSet)
{
  bool result = true;
  bool intermediate;
  if ((GreaterSet < 0) || (SmallerSet < 0) || (GreaterSet > FragmentCounter) || (SmallerSet > FragmentCounter)) // index out of bounds
    return false;
  for(int i=AtomCounter[SmallerSet];i--;) {
    intermediate = false;
    for (int j=AtomCounter[GreaterSet];j--;)
      intermediate = (intermediate || ((KeySets[SmallerSet][i] == KeySets[GreaterSet][j]) || (KeySets[SmallerSet][i] == -1)));
    result = result && intermediate;
  }

  return result;
};

