/*
 * Shape_impl.hpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

#ifndef SHAPE_IMPL_HPP_
#define SHAPE_IMPL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <limits>
#include <vector>

#include "CodePatterns/Assert.hpp"

#include "Shapes/Shape.hpp"
#include "Shapes/ShapeExceptions.hpp"
#include "Shapes/ShapeType.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/LinePoint.hpp"
#include "LinearAlgebra/LineSegment.hpp"
#include "LinearAlgebra/LineSegmentSet.hpp"
#include "LinearAlgebra/Vector.hpp"


class Shape_impl {
public:
  Shape_impl(){};
  virtual ~Shape_impl(){};
  virtual bool isInside(const Vector &point) const=0;
  virtual bool isOnSurface(const Vector &point) const=0;
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException)=0;
  virtual Vector getCenter() const=0;
  virtual double getRadius() const=0;
  virtual LineSegmentSet getLineIntersections(const Line&) const=0;
  virtual std::string toString() const =0;
  virtual enum ShapeType getType() const =0;
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const=0;
  virtual   std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const=0;
};

class Everywhere_impl : public Shape_impl {
public:
  virtual bool isInside(const Vector &point) const{
    return true;
  }
  virtual bool isOnSurface(const Vector &point) const{
    return false;
  }
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException){
    throw NotOnSurfaceException() << ShapeVector(&point);
  }
  virtual Vector getCenter() const {
    return Vector(0.,0.,0.);
  }
  virtual double getRadius() const {
    return std::numeric_limits<double>::infinity();
  }
  virtual LineSegmentSet getLineIntersections(const Line &line) const{
    LineSegmentSet res(line);
    res.insert(LineSegment(line.negEndpoint(),line.posEndpoint()));
    return res;
  }
  virtual std::string toString() const{
    return "Everywhere()";
  }
  virtual enum ShapeType getType() const {
  	return EverywhereType;
  }
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const {
    std::vector<Vector> PointsOnSurface;
    return PointsOnSurface;
  }
  std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const {
  	ASSERT(0,
  			"Everywhere_impl::getHomogeneousPointsInVolume() - not implemented.");
  	return std::vector<Vector>();
  }
};

class Nowhere_impl : public Shape_impl {
  virtual bool isInside(const Vector &point) const{
    return false;
  }
  virtual bool isOnSurface(const Vector &point) const{
    return false;
  }
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException){
    throw NotOnSurfaceException() << ShapeVector(&point);
  }
  virtual Vector getCenter() const {
    return Vector(0.,0.,0.);
  }
  virtual double getRadius() const {
    return 0.;
  }
  virtual LineSegmentSet getLineIntersections(const Line &line) const{
    return LineSegmentSet(line);
  }
  virtual std::string toString() const{
    return "Nowhere()";
  }
  virtual enum ShapeType getType() const {
  	return NowhereType;
  }
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const {
    std::vector<Vector> PointsOnSurface;
    return PointsOnSurface;
  }
  std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const {
  	return std::vector<Vector>();
  }
};

class AndShape_impl : public Shape_impl {
public:
  AndShape_impl(const Shape::impl_ptr&, const Shape::impl_ptr&);
  virtual ~AndShape_impl();
  virtual bool isInside(const Vector &point) const;
  virtual bool isOnSurface(const Vector &point) const;
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException);
  virtual Vector getCenter() const;
  virtual double getRadius() const;
  virtual LineSegmentSet getLineIntersections(const Line&) const;
  virtual std::string toString() const;
  virtual enum ShapeType getType() const;
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const;
  virtual std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const;
private:
  Shape::impl_ptr lhs;
  Shape::impl_ptr rhs;
};

class OrShape_impl : public Shape_impl {
public:
  OrShape_impl(const Shape::impl_ptr&, const Shape::impl_ptr&);
  virtual ~OrShape_impl();
  virtual bool isInside(const Vector &point) const;
  virtual bool isOnSurface(const Vector &point) const;
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException);
  virtual Vector getCenter() const;
  virtual double getRadius() const;
  virtual LineSegmentSet getLineIntersections(const Line&) const;
  virtual std::string toString() const;
  virtual enum ShapeType getType() const;
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const;
  virtual std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const;
private:
  Shape::impl_ptr lhs;
  Shape::impl_ptr rhs;
};

class NotShape_impl : public Shape_impl {
public:
  NotShape_impl(const Shape::impl_ptr&);
  virtual ~NotShape_impl();
  virtual bool isInside(const Vector &point) const;
  virtual bool isOnSurface(const Vector &point) const;
  virtual Vector getNormal(const Vector &point) const throw(NotOnSurfaceException);
  virtual Vector getCenter() const;
  virtual double getRadius() const;
  virtual LineSegmentSet getLineIntersections(const Line&) const;
  virtual std::string toString() const;
  virtual enum ShapeType getType() const;
  virtual std::vector<Vector> getHomogeneousPointsOnSurface(const size_t N) const;
  virtual std::vector<Vector> getHomogeneousPointsInVolume(const size_t N) const;
private:
  Shape::impl_ptr arg;
};

Shape::impl_ptr getShapeImpl(const Shape&);

#endif /* SHAPE_IMPL_HPP_ */
