/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AddEmptyBoundaryAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include "boost/serialization/vector.hpp"

//#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Box.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/MatrixContent.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Actions/WorldAction/AddEmptyBoundaryAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AddEmptyBoundaryAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr WorldAddEmptyBoundaryAction::performCall() {
  Vector Min;
  Vector Max;
  int j=0;

  // create undo domain
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  const RealSpaceMatrix &matrix = World::getInstance().getDomain().getM();
  oa << matrix;

  // get maximum and minimum
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  if (AllAtoms.size() > 0) {
    std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
    Min = (*AtomRunner)->getPosition();
    Max = (*AtomRunner)->getPosition();
    for (; AtomRunner != AllAtoms.end(); ++AtomRunner) {
      for (int i=0;i<NDIM;i++) {
        if ((*AtomRunner)->at(i) > Max[i])
          Max[i] = (*AtomRunner)->at(i);
        if ((*AtomRunner)->at(i) < Min[i])
          Min[i] = (*AtomRunner)->at(i);
      }
    }
    // set new box size
    double * const cell_size = new double[6];
    for (j=0;j<6;j++)
      cell_size[j] = 0.;
    j=-1;
    for (int i=0;i<NDIM;i++) {
      j += i+1;
      cell_size[j] = (Max[i]-Min[i]+2.*params.boundary.get()[i]);
    }
    World::getInstance().setDomain(cell_size);
    delete[] cell_size;
    // translate all atoms, such that Min is aty (0,0,0)
    AtomRunner = AllAtoms.begin();
    for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
        AtomRunner != AllAtoms.end();
        ++AtomRunner)
      *(*AtomRunner) -= Min - params.boundary.get();

    // give final box size
    LOG(0, "Box domain is now " << World::getInstance().getDomain().getM());

    // create undo state
    WorldAddEmptyBoundaryState *UndoState =
        new WorldAddEmptyBoundaryState(
            undostream.str(),
            World::getInstance().getDomain().getM(),
            Min,
            params
            );

    return ActionState::ptr(UndoState);
  } else {
    STATUS("There must be atoms present for AddingEmptyBoundary.");
    return Action::failure;
  }
}

ActionState::ptr WorldAddEmptyBoundaryAction::performUndo(ActionState::ptr _state) {
  WorldAddEmptyBoundaryState *state = assert_cast<WorldAddEmptyBoundaryState*>(_state.get());

  // restore domain
  RealSpaceMatrix matrix;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> matrix;
  World::getInstance().setDomain(matrix);

  // give final box size
  LOG(0, "Box domain restored to " << World::getInstance().getDomain().getM());

  // restore atoms
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) += state->Min - state->params.boundary.get();

  return ActionState::ptr(_state);
}

ActionState::ptr WorldAddEmptyBoundaryAction::performRedo(ActionState::ptr _state){
  WorldAddEmptyBoundaryState *state = assert_cast<WorldAddEmptyBoundaryState*>(_state.get());

  World::getInstance().setDomain(state->newdomain);

  // give final box size
  LOG(0, "Box domain is again " << World::getInstance().getDomain().getM());

  // shift atoms
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) -= state->Min - state->params.boundary.get();

  return ActionState::ptr(_state);
}

bool WorldAddEmptyBoundaryAction::canUndo() {
  return true;
}

bool WorldAddEmptyBoundaryAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
