/*
 * SphericalPointDistribution.hpp
 *
 *  Created on: May 29, 2014
 *      Author: heber
 */


#ifndef SPHERICALPOINTDISTRIBUTION_HPP_
#define SPHERICALPOINTDISTRIBUTION_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Assert.hpp"

#include <cmath>
#include <list>

#include "LinearAlgebra/Vector.hpp"

class SphericalPointDistributionTest;

/** contains getters for the VSEPR model for specific number of electrons.
 *
 * This struct contains specialized functions returning a list of Vectors
 * (points in space) to match the VSEPR model for the given number of electrons.
 *
 * This is implemented via template specialization of the function get().
 *
 * These specializations are taken from the python script \b CreateVspeShapes.py
 * by Christian Neuen, 07th May 2009.
 */
struct SphericalPointDistribution
{
  /** Cstor for SphericalPointDistribution, allows setting radius of sphere
   *
   * \param _BondLength desired radius of sphere
   */
  SphericalPointDistribution(const double _Bondlength = 1.) :
    Bondlength(_Bondlength)
  {}

  //!> typedef for the list of points
  typedef std::list<Vector> Polygon_t;
  //!> typedef for the list of points with integral weights
  typedef std::list<std::pair<Vector, int> > WeightedPolygon_t;

  /** General getter function for the distribution of points on the surface.
   *
   * \warn this function needs to be specialized!
   *
   * \return Polygon_t with points on the surface centered at (0,0,0)
   */
  template <int N> Polygon_t get() const
  {
    ASSERT(0, "SphericalPointDistribution::get() - not specialized for "+toString(N)+".");
    return Polygon_t();
  }

  /** Matches a given spherical distribution with another containing more
   * points.
   *
   * This is a helper to determine points where to best insert saturation
   * hydrogens.
   *
   * \param _polygon current occupied positions
   * \param _newpolygon ideal distribution to match best with current occupied
   *        positions
   * \return remaining vacant positions relative to \a _polygon
   */
  static Polygon_t matchSphericalPointDistributions(
      const WeightedPolygon_t &_polygon,
      Polygon_t &_newpolygon
      );

  //!> default radius of the spherical distribution
  const double Bondlength;
  //!> precalculated value for root of 3
  static const double SQRT_3;
  //!> threshold for L1 error below which matching is immediately acceptable
  static const double L1THRESHOLD;
  //!> threshold for L2 error below which matching is acceptable
  static const double L2THRESHOLD;

  //!> typedef for a full rotation specification consisting of axis and angle.
  typedef std::pair<Vector, double> Rotation_t;

  //!> typedef for a list of indices (of points in a polygon)
  typedef std::list<unsigned int> IndexList_t;
  //!> typedef enumerating possibly multiple points accumulated as one point
  typedef std::list< IndexList_t > IndexTupleList_t;
  //!> typedef for a vector of indices
  typedef std::vector<unsigned int> IndexArray_t;
  //!> typedef for a Vector of positions
  typedef std::vector<Vector> VectorArray_t;
  //!> typedef for a Vector of positions with weights
  typedef std::vector< std::pair<Vector, int> > WeightedVectorArray_t;
  //!> typedef for a vector of degrees (or integral weights)
  typedef std::vector<unsigned int> WeightsArray_t;

  //!> amplitude up to which deviations in checks of rotations are tolerated
  static const double warn_amplitude;

private:
  //!> grant unit tests access to private parts
  friend class SphericalPointDistributionTest;

  static std::pair<double, double> calculateErrorOfMatching(
      const VectorArray_t &_old,
      const VectorArray_t &_new,
      const IndexTupleList_t &_Matching);

  static Polygon_t removeMatchingPoints(
      const VectorArray_t &_points,
      const IndexList_t &_matchingindices
      );

  struct MatchingControlStructure {
    bool foundflag;
    double bestL2;
    IndexTupleList_t bestmatching;
    VectorArray_t oldpoints;
    VectorArray_t newpoints;
    WeightsArray_t weights;
  };

  static void recurseMatchings(
      MatchingControlStructure &_MCS,
      IndexTupleList_t &_matching,
      IndexList_t _indices,
      WeightsArray_t &_remainingweights,
      WeightsArray_t::iterator _remainiter,
      const unsigned int _matchingsize
      );

  static IndexList_t findBestMatching(
      const WeightedPolygon_t &_polygon,
      Polygon_t &_newpolygon
      );

  static IndexList_t joinPoints(
      Polygon_t &_newpolygon,
      const VectorArray_t &_newpoints,
      const IndexTupleList_t &_bestmatching
      );

  static Rotation_t findPlaneAligningRotation(
      const VectorArray_t &_referencepositions,
      const VectorArray_t &_currentpositions,
      const IndexList_t &_bestmatching
      );

  static Rotation_t findPointAligningRotation(
      const VectorArray_t &remainingold,
      const VectorArray_t &remainingnew,
      const IndexList_t &_bestmatching);

};

// declare specializations

template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<0>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<1>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<2>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<3>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<4>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<5>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<6>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<7>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<8>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<9>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<10>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<11>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<12>() const;
template <> SphericalPointDistribution::Polygon_t SphericalPointDistribution::get<14>() const;

#endif /* SPHERICALPOINTDISTRIBUTION_HPP_ */
