/*
 * Action.h
 *
 *  Created on: Dec 8, 2009
 *      Author: crueger
 */

#ifndef ACTION_H_
#define ACTION_H_

#include <string>
#include <boost/shared_ptr.hpp>

// forward declaration

class ActionState;
class ActionSequence;

/**
 * Base class for all actions.
 *
 * Actions describe something that has to be done.
 * Actions can be passed around, stored, performed and undone (Command-Pattern).
 *
 * TODO: Add queues of actions that have been performed to allow easy implementation of multiple-step undo
 */
class Action
{
friend class ActionSequence;
public:

  typedef boost::shared_ptr<ActionState> state_ptr;

  Action(std::string _name,bool _doRegister=true);
  virtual ~Action();

  // this method only handles the infrastructure
  // actuall action is passed on to a private virtual
  void call();
  state_ptr undo(state_ptr);
  state_ptr redo(state_ptr);

  virtual bool canUndo()=0;
  virtual bool shouldUndo()=0;

  virtual const std::string getName();

protected:
  static state_ptr success;
  static state_ptr failure;

private:
  virtual state_ptr performCall()=0;
  virtual state_ptr performUndo(state_ptr)=0;
  virtual state_ptr performRedo(state_ptr)=0;

  std::string name;
};

/**
 * This class can be used by actions to save the state.
 *
 * It is implementing a memento pattern. The base class is completely empty,
 * since no general state internals can be given. The Action performing
 * the Undo should downcast to the apropriate type.
 */
class ActionState{
public:
  ActionState(){}
  virtual ~ActionState(){}
};

#endif /* ACTION_H_ */
