/*
 * bondgraph.cpp
 *
 *  Created on: Oct 29, 2009
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include <iostream>

#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "info.hpp"
#include "verbose.hpp"
#include "log.hpp"
#include "molecule.hpp"
#include "parser.hpp"
#include "periodentafel.hpp"
#include "vector.hpp"

/** Constructor of class BondGraph.
 * This classes contains typical bond lengths and thus may be used to construct a bond graph for a given molecule.
 */
BondGraph::BondGraph(bool IsA) : BondLengthMatrix(NULL), max_distance(0), IsAngstroem(IsA)
{
};

/** Destructor of class BondGraph.
 */
BondGraph::~BondGraph()
{
  if (BondLengthMatrix != NULL) {
    delete(BondLengthMatrix);
  }
};

/** Parses the bond lengths in a given file and puts them int a matrix form.
 * Allocates \a MatrixContainer for BondGraph::BondLengthMatrix, using MatrixContainer::ParseMatrix(),
 * but only if parsing is successful. Otherwise variable is left as NULL.
 * \param *out output stream for debugging
 * \param filename file with bond lengths to parse
 * \return true - success in parsing file, false - failed to parse the file
 */
bool BondGraph::LoadBondLengthTable(const string &filename)
{
  Info FunctionInfo(__func__);
  bool status = true;
  MatrixContainer *TempContainer = NULL;

  // allocate MatrixContainer
  if (BondLengthMatrix != NULL) {
    DoLog(1) && (Log() << Verbose(1) << "MatrixContainer for Bond length already present, removing." << endl);
    delete(BondLengthMatrix);
  }
  TempContainer = new MatrixContainer;

  // parse in matrix
  if ((status = TempContainer->ParseMatrix(filename.c_str(), 0, 1, 0))) {
    DoLog(1) && (Log() << Verbose(1) << "Parsing bond length matrix successful." << endl);
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "Parsing bond length matrix failed." << endl);
  }

  // find greatest distance
  max_distance=0;
  if (status) {
    for(int i=0;i<TempContainer->RowCounter[0];i++)
      for(int j=i;j<TempContainer->ColumnCounter[0];j++)
        if (TempContainer->Matrix[0][i][j] > max_distance)
          max_distance = TempContainer->Matrix[0][i][j];
  }

  if (status) // set to not NULL only if matrix was parsed
    BondLengthMatrix = TempContainer;
  else {
    BondLengthMatrix = NULL;
    delete(TempContainer);
  }
  return status;
};

/** Parses the bond lengths in a given file and puts them int a matrix form.
 * \param *out output stream for debugging
 * \param *mol molecule with atoms
 * \return true - success, false - failed to construct bond structure
 */
bool BondGraph::ConstructBondGraph(molecule * const mol)
{
  Info FunctionInfo(__func__);
  bool status = true;

  if (mol->empty()) // only construct if molecule is not empty
    return false;

  if (BondLengthMatrix == NULL) { // no bond length matrix parsed?
    SetMaxDistanceToMaxOfCovalentRadii(mol);
    mol->CreateAdjacencyList(max_distance, IsAngstroem, &BondGraph::CovalentMinMaxDistance, this);
  } else
    mol->CreateAdjacencyList(max_distance, IsAngstroem, &BondGraph::BondLengthMatrixMinMaxDistance, this);

  return status;
};

/** Returns the entry for a given index pair.
 * \param firstelement index/atom number of first element (row index)
 * \param secondelement index/atom number of second element (column index)
 * \note matrix is of course symmetric.
 */
double BondGraph::GetBondLength(int firstZ, int secondZ)
{
  if (BondLengthMatrix == NULL)
    return( -1. );
  else
    return (BondLengthMatrix->Matrix[0][firstZ][secondZ]);
};

/** Determines the maximum of all element::CovalentRadius for elements present in \a *mol.
 * \param *out output stream for debugging
 * \param *mol molecule with all atoms and their respective elements.
 */
double BondGraph::SetMaxDistanceToMaxOfCovalentRadii(const molecule * const mol)
{
  Info FunctionInfo(__func__);
  max_distance = 0.;

  for (molecule::const_iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    if ((*iter)->type->CovalentRadius > max_distance)
      max_distance = (*iter)->type->CovalentRadius;
  }
  max_distance *= 2.;

  return max_distance;
};

/** Returns bond criterion for given pair based on covalent radius.
 * \param *Walker first BondedParticle
 * \param *OtherWalker second BondedParticle
 * \param &MinDistance lower bond bound on return
 * \param &MaxDistance upper bond bound on return
 * \param IsAngstroem whether units are in angstroem or bohr radii
 */
void BondGraph::CovalentMinMaxDistance(BondedParticle * const Walker, BondedParticle * const OtherWalker, double &MinDistance, double &MaxDistance, bool IsAngstroem)
{
  MinDistance = OtherWalker->type->CovalentRadius + Walker->type->CovalentRadius;
  MinDistance *= (IsAngstroem) ? 1. : 1. / AtomicLengthToAngstroem;
  MaxDistance = MinDistance + BONDTHRESHOLD;
  MinDistance -= BONDTHRESHOLD;
};

/** Returns bond criterion for given pair based on a bond length matrix.
 * The matrix should be contained in \a this BondGraph and contain an element-
 * to-element length.
 * \param *Walker first BondedParticle
 * \param *OtherWalker second BondedParticle
 * \param &MinDistance lower bond bound on return
 * \param &MaxDistance upper bond bound on return
 * \param IsAngstroem whether units are in angstroem or bohr radii
 */
void BondGraph::BondLengthMatrixMinMaxDistance(BondedParticle * const Walker, BondedParticle * const OtherWalker, double &MinDistance, double &MaxDistance, bool IsAngstroem)
{
  if (BondLengthMatrix == NULL) {// safety measure if no matrix has been parsed yet
    DoeLog(2) && (eLog()<< Verbose(2) << "BondLengthMatrixMinMaxDistance() called without having parsed the bond length matrix yet!" << endl);
    CovalentMinMaxDistance(Walker, OtherWalker, MinDistance, MaxDistance, IsAngstroem);
  } else {
    MinDistance = GetBondLength(Walker->type->Z-1, OtherWalker->type->Z-1);
    MinDistance *= (IsAngstroem) ? 1. : 1. / AtomicLengthToAngstroem;
    MaxDistance = MinDistance + BONDTHRESHOLD;
    MinDistance -= BONDTHRESHOLD;
  }
};
