/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtInfoBox.cpp
 *
 *  Created on: Mar 4, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtInfoBox.hpp"

#include <iostream>
#include <QtGui/QAbstractItemView>

//#include "CodePatterns/MemDebug.hpp"

#include "UIElements/Qt4/InstanceBoard/QtObservedInstanceBoard.hpp"

#include <boost/thread/locks.hpp>

#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "World.hpp"

using namespace std;

/***************** Basic structure for tab layout ***********/

QtInfoBox::QtInfoBox(QtObservedInstanceBoard *_board, QWidget *_parent) :
    QTabWidget(_parent),
    curAtomId(-1), nextAtomId(-1),
    curMoleculeId(-1), nextMoleculeId(-1),
    page_mol(NULL), page_atom(NULL),
    board(_board),
    currentPage(0),
    periode(World::getInstance().getPeriode())
{
  timer = new QTimer(this);
  timer->setSingleShot(true);

  setMinimumWidth(200);
  setMinimumHeight(350);

  connect(timer, SIGNAL(timeout()), this, SLOT(timerTimeout()));
}

QtInfoBox::~QtInfoBox()
{
  clearAtomTab();
  clearMoleculeTab();
}

void QtInfoBox::atomHover(const atomId_t _id)
{
  nextAtomId = _id;
  timer->start(500);
}

void QtInfoBox::moleculeHover(const moleculeId_t _id)
{
  nextMoleculeId = _id;
  timer->start(500);
}

void QtInfoBox::timerTimeout()
{
  if (nextAtomId != (atomId_t)-1)
    showAtom(nextAtomId);
  if (nextMoleculeId != (moleculeId_t)-1)
    showMolecule(nextMoleculeId);
}

void QtInfoBox::clearAtomTab()
{
  boost::recursive_mutex::scoped_lock lock(tabs_mutex);
  if (page_atom){
    if (page_atom->getAtom())
      disconnect(page_atom->getAtom().get(), SIGNAL(atomRemoved()), this, SLOT(clearAtomTab()));
    //removeTab(indexOf(page_atom));
    delete(page_atom);
    page_atom = NULL;
  }
}

void QtInfoBox::clearMoleculeTab()
{
  boost::recursive_mutex::scoped_lock lock(tabs_mutex);
  if (page_mol){
    if (page_mol->getMolecule())
      disconnect(page_mol->getMolecule().get(), SIGNAL(moleculeRemoved()), this, SLOT(clearMoleculeTab()));
    //removeTab(indexOf(page_mol));
    delete(page_mol);
    page_mol = NULL;
  }
}

void QtInfoBox::showAtom(const atomId_t _id)
{
  currentPage = currentIndex();

  // Remove old tabs.
  clearAtomTab();
  clearMoleculeTab();

  QtObservedAtom::ptr curAtom = board->getObservedAtom(_id);

  // Show new tabs.
  if (curAtom){
    curAtomId = curAtom->getAtomIndex();
    nextAtomId = -1;
    nextMoleculeId = -1;

    boost::recursive_mutex::scoped_lock lock(tabs_mutex);
    page_atom = new QtAtomInfoPage(curAtom, periode, this);
    addTab(page_atom, "Atom");
    connect(curAtom.get(), SIGNAL(atomRemoved()), this, SLOT(clearAtomTab()));

    if (curAtom->getMoleculeRef() != NULL) {
      QtObservedMolecule::ptr curMolecule = curAtom->getMoleculeRef()->getRef();
      if (curMolecule) {
        page_mol = new QtMoleculeInfoPage(curMolecule, this);
        addTab(page_mol, "Molecule");
        connect(curMolecule.get(), SIGNAL(moleculeRemoved()), this, SLOT(clearMoleculeTab()));

        if (currentPage > 0)
          setCurrentIndex(currentPage);
      }
    }
  }
}

void QtInfoBox::showMolecule(const moleculeId_t _id)
{
  currentPage = currentIndex();

  // Remove old tabs.
  clearAtomTab();
  clearMoleculeTab();

  QtObservedMolecule::ptr curMolecule = board->getObservedMolecule(_id);
  nextAtomId = -1;
  nextMoleculeId = -1;

  // Show new tabs.
  if (curMolecule){
    boost::recursive_mutex::scoped_lock lock(tabs_mutex);
    page_mol = new QtMoleculeInfoPage(curMolecule, this);
    addTab(page_mol, "Molecule");
    connect(curMolecule.get(), SIGNAL(moleculeRemoved()), this, SLOT(clearMoleculeTab()));

    if (currentPage > 0)
      setCurrentIndex(currentPage);
  }
}

/************************ Tab for single Atoms ********************/

static void addInfo(QTreeWidget *info, const QString &key, const QString &value)
{
  QTreeWidgetItem *treeItem = new QTreeWidgetItem(info);
  treeItem->setText(0, key);
  treeItem->setText(1, value);
}

QtAtomInfoPage::QtAtomInfoPage(
    QtObservedAtom::ptr &_atom,
    periodentafel *_periode,
    QWidget *parent) :
  QTreeWidget(parent),
  atomRef(_atom),
  periode(_periode)
{
  setColumnCount(2);
  QStringList header;
  header << "data";
  header << "value";
  setHeaderLabels(header);

  updatePage();
}

void QtAtomInfoPage::updatePage()
{
  clear();

  if (atomRef == NULL)
    return;
  const element * const _element = periode->FindElement(atomRef->getAtomElement());
  addInfo(this, "Name", QString(atomRef->getAtomName().c_str()));
  addInfo(this, "Element", QString(_element->getName().c_str()));
  addInfo(this, "Mass", QString("%1").arg(_element->getMass()));
  addInfo(this, "Charge", QString("%1").arg(_element->getCharge()));
  addInfo(this, "Bonds", QString("%1").arg(atomRef->getAtomBonds().size()));
  addInfo(this, "Position x", QString(toString(atomRef->getAtomPosition()[0]).c_str()));
  addInfo(this, "Position y", QString(toString(atomRef->getAtomPosition()[1]).c_str()));
  addInfo(this, "Position z", QString(toString(atomRef->getAtomPosition()[2]).c_str()));
  addInfo(this, "Velocity x", QString(toString(atomRef->getAtomVelocity()[0]).c_str()));
  addInfo(this, "Velocity y", QString(toString(atomRef->getAtomVelocity()[1]).c_str()));
  addInfo(this, "Velocity z", QString(toString(atomRef->getAtomVelocity()[2]).c_str()));
  addInfo(this, "Force x", QString(toString(atomRef->getAtomForce()[0]).c_str()));
  addInfo(this, "Force y", QString(toString(atomRef->getAtomForce()[1]).c_str()));
  addInfo(this, "Force z", QString(toString(atomRef->getAtomForce()[2]).c_str()));
}

QtAtomInfoPage::~QtAtomInfoPage()
{}

/************************ Tab for single Molecules *****************/

QtMoleculeInfoPage::QtMoleculeInfoPage(
    QtObservedMolecule::ptr &_mol,
    QWidget *parent) :
    QTreeWidget(parent),
    mol(_mol)
{
  setColumnCount(2);
  QStringList header;
  header << "data";
  header << "value";
  setHeaderLabels(header);

  updatePage();
}

void QtMoleculeInfoPage::updatePage()
{
  clear();

  if (mol == NULL)
    return;
  addInfo(this, "Name", QString(mol->getMolName().c_str()));
  addInfo(this, "Formula", QString(mol->getMolFormula().c_str()));
  addInfo(this, "Atoms", QString("%1").arg(mol->getAtomCount()));
  addInfo(this, "NonHydrogens", QString("%1").arg(mol->getNonHydrogenCount()));
  addInfo(this, "Bonds", QString("%1").arg(mol->getBondCount()));
  const Vector molCenter = mol->getMolCenter();
  addInfo(this, "Center x", QString("%1").arg(molCenter[0]));
  addInfo(this, "Center y", QString("%1").arg(molCenter[1]));
  addInfo(this, "Center z", QString("%1").arg(molCenter[2]));
}

QtMoleculeInfoPage::~QtMoleculeInfoPage()
{}
