/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * BondLengthTableAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Graph/BondGraph.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

#include "Actions/CommandAction/BondLengthTableAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "BondLengthTableAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr CommandBondLengthTableAction::performCall() {
  ostringstream usage;

  // obtain information
  getParametersfromValueStorage();

  LOG(0, "Using " << params.BondGraphFileName << " as bond length table.");
  BondGraph *&BG = World::getInstance().getBondGraph();

  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  oa << BG;
  CommandBondLengthTableState *UndoState =
      new CommandBondLengthTableState(
          undostream.str(),
          params
      );

  BG->CleanupBondLengthTable();
  if ((!params.BondGraphFileName.empty())
      && boost::filesystem::exists(params.BondGraphFileName)) {
    std::ifstream input(params.BondGraphFileName.string().c_str());
    if ((input.good()) && (BG->LoadBondLengthTable(input))) {
      LOG(0, "Bond length table parsed successfully.");
      input.close();
      return Action::state_ptr(UndoState);
    } else {
      ELOG(1, "Bond length table parsing failed.");
      input.close();
    }
  } else {
    ELOG(1, "Bond length table loading failed.");
  }
  // recover bond graph
  boost::archive::text_iarchive ia(undostream);
  delete BG;
  ia >> BG;
  delete UndoState;
  return Action::failure;
}

Action::state_ptr CommandBondLengthTableAction::performUndo(Action::state_ptr _state) {
  CommandBondLengthTableState *state = assert_cast<CommandBondLengthTableState*>(_state.get());

  BondGraph *BG;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> BG;
  World::getInstance().setBondGraph(BG);

  return Action::state_ptr(_state);
}

Action::state_ptr CommandBondLengthTableAction::performRedo(Action::state_ptr _state){
  CommandBondLengthTableState *state = assert_cast<CommandBondLengthTableState*>(_state.get());

  BondGraph *&BG = World::getInstance().getBondGraph();
  BG->CleanupBondLengthTable();
  std::ifstream input(state->params.BondGraphFileName.string().c_str());
  if ((input.good()) && (BG->LoadBondLengthTable(input))) {
    LOG(0, "Bond length table parsed successfully.");
    input.close();
  }

  return Action::state_ptr(_state);
}

bool CommandBondLengthTableAction::canUndo() {
  return true;
}

bool CommandBondLengthTableAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
