/*
 * ConstantPotential.hpp
 *
 *  Created on: May 09, 2013
 *      Author: heber
 */

#ifndef CONSTANTPOTENTIAL_HPP_
#define CONSTANTPOTENTIAL_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits>

#include "Potentials/EmpiricalPotential.hpp"

class PotentialFactory;
class TrainingData;

/** This is the implementation of a constant potential to adapt to any offset
 * in the potential energy.
 *
 * This evaluates \f$ k \f$.
 *
 */
class ConstantPotential :
  public EmpiricalPotential
{
  //!> grant unit test access to internal parts
  friend class ConstantPotentialTest;
  //!> grant PotentialFactory access to default cstor
  friend class PotentialFactory;
  // some repeated typedefs to avoid ambiguities
  typedef FunctionModel::list_of_arguments_t list_of_arguments_t;
  typedef FunctionModel::arguments_t arguments_t;
  typedef FunctionModel::result_t result_t;
  typedef FunctionModel::results_t results_t;
  typedef EmpiricalPotential::derivative_components_t derivative_components_t;
  typedef FunctionModel::parameters_t parameters_t;
private:
  /** Private default constructor.
   *
   * This prevents creation of potential without set ParticleTypes_t.
   *
   * \note PotentialFactory may use this default cstor
   *
   */
  ConstantPotential();

public:
  ConstantPotential(const ParticleTypes_t &_ParticleTypes);
  ConstantPotential(
      const ParticleTypes_t &_ParticleTypes,
      const double _energy_offset);
  virtual ~ConstantPotential() {}

  /** Setter for parameters as required by FunctionModel interface.
   *
   * \param _params given set of parameters
   */
  void setParameters(const parameters_t &_params);

  /** Getter for parameters as required by FunctionModel interface.
   *
   * \return set of parameters
   */
  parameters_t getParameters() const
  {
    return params;
  }

  /** Sets the parameter randomly within the sensible range of each parameter.
   *
   * \param data container with training data for guesstimating range
   */
  void setParametersToRandomInitialValues(const TrainingData &data);

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  size_t getParameterDimension() const
  {
    return 1;
  }

  /** Evaluates the harmonic potential function for the given arguments.
   *
   * @param listarguments empty list
   * @return value of the potential function
   */
  results_t operator()(const list_of_arguments_t &listarguments) const;

  /** Evaluates the derivative of the potential function.
   *
   * @param listarguments empty list
   * @return vector with derivative with respect to the input degrees of freedom
   */
  derivative_components_t derivative(const list_of_arguments_t &listarguments) const;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param listarguments empty list
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  results_t parameter_derivative(const list_of_arguments_t &listarguments, const size_t index) const;

  /** Returns the functor that converts argument_s into the
   * internal coordinate described by this potential function.
   *
   * \return coordinator functor
   */
  Coordinator::ptr getCoordinator() const
  { return coordinator; }

  /** Return the token name of this specific potential.
   *
   * \return token name of the potential
   */
  const std::string& getToken() const
  { return potential_token; }

  /** Returns a vector of parameter names.
   *
   * This is required from the specific implementation
   *
   * \return vector of strings containing parameter names
   */
  const ParameterNames_t& getParameterNames() const
  { return ParameterNames; }

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  bool isBoxConstraint() const {
    return true;
  }

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  parameters_t getLowerBoxConstraints() const {
    parameters_t lowerbound(getParameterDimension(), -std::numeric_limits<double>::max());
    return lowerbound;
  }

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  parameters_t getUpperBoxConstraints() const {
    return parameters_t(getParameterDimension(), std::numeric_limits<double>::max());
  }

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \return bound function extracting distances from a fragment
   */
  FunctionModel::filter_t getSpecificFilter() const;

  /** Returns the number of arguments the underlying function requires.
   *
   * \return number of arguments of the function
   */
  size_t getSpecificArgumentCount() const
  { return 0; }

  enum parameter_enum_t {
    energy_offset=0,
    MAXPARAMS
  };

  /** Getter for the graph specifying the binding model of the potential.
   *
   * \return BindingModel ref of the binding model
   */
  const BindingModel& getBindingModel() const
  { return bindingmodel; }

  /**
   * Returns the number of particle types associated with the potential.
   *
   * \return number of particle types
   */
  unsigned int getParticleTypeNumber() const
  { return 0; }

private:
  //!> parameter vector with parameters as in enum parameter_enum_t
  parameters_t params;

  //!> static definitions of the parameter name for this potential
  static const ParameterNames_t ParameterNames;

  //!> static token of this potential type
  static const std::string potential_token;

  //!> internal coordinator object for converting arguments_t
  static Coordinator::ptr coordinator;

  //!> binding model for this potential
  const BindingModel bindingmodel;
};

#endif /* CONSTANTPOTENTIAL_HPP_ */
