/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CheckAgainstAdjacencyFileUnitTest.cpp
 *
 *  Created on: Oct 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CheckAgainstAdjacencyFileUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <string>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Descriptors/AtomDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Graph/CheckAgainstAdjacencyFile.hpp"
#include "molecule.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( CheckAgainstAdjacencyFileTest );

static std::string adjacencyfile ="\
1 2\n\
2 1 3\n\
3 2 4\n\
4 3 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9\n";

static std::string wrongadjacencyfile1 ="\
1 2\n\
2 1 3\n\
4 3 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9\n";

static std::string wrongadjacencyfile2 ="\
1 2\n\
2 1 3\n\
3 2 4\n\
4 3 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9 11\n\
11 10";

// set up and tear down
void CheckAgainstAdjacencyFileTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  const element *hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL);

  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL);
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    CPPUNIT_ASSERT(atoms[i] != NULL);
    atoms[i]->setType(hydrogen);
    TestMolecule->AddAtom(atoms[i]);
    atomIds[i]= atoms[i]->getId();
  }
  // create linear chain
  for(int i=0;i<ATOM_COUNT-1;++i){
    atoms[i]->addBond(WorldTime::getTime(), atoms[i+1]);
  }

  // create map as it should be
  for(int i=0;i<ATOM_COUNT;++i) {
    if (i != 0) // first has only one bond
      comparisonMap.insert( std::make_pair(atomIds[i], atomIds[i-1]) );
    if (i != ATOM_COUNT-1) // last has only one bond
      comparisonMap.insert( std::make_pair(atomIds[i], atomIds[i+1]) );
  }
}

void CheckAgainstAdjacencyFileTest::tearDown()
{
  comparisonMap.clear();

  // destroy molecule and contained atoms
  TestMolecule->removeAtomsinMolecule();
  World::getInstance().destroyMolecule(TestMolecule);
  // destroy World
  World::purgeInstance();
//  logger::purgeInstance();
//  errorLogger::purgeInstance();
}

/** Unit tests for CheckAgainstAdjacencyFile::CreateInternalMap().
 *
 */
void CheckAgainstAdjacencyFileTest::CreateInternalMapTest()
{
  std::stringstream input;
  CheckAgainstAdjacencyFile fileChecker(input);
  World::getInstance().selectAllAtoms(AllAtoms());
  const World &world = World::getInstance();
  fileChecker.CreateInternalMap(world.beginAtomSelection(), world.endAtomSelection());

  // check size (it's 8*2 + 2*1 = 18 keys)
  CPPUNIT_ASSERT_EQUAL( (size_t)18, fileChecker.InternalAtomBondMap.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, fileChecker.ExternalAtomBondMap.size() );

  // check equality
  CPPUNIT_ASSERT( comparisonMap.size() == fileChecker.InternalAtomBondMap.size() );
//  std::cout << "comparisonMap: " << comparisonMap << std::endl;
//  std::cout << "fileChecker.InternalAtomBondMap: " << fileChecker.InternalAtomBondMap << std::endl;
  CPPUNIT_ASSERT( comparisonMap == fileChecker.InternalAtomBondMap );

  // check non-equality: more
  comparisonMap.insert( std::make_pair( (atomId_t)10, (atomId_t)8) );
  CPPUNIT_ASSERT( comparisonMap != fileChecker.InternalAtomBondMap );
  comparisonMap.erase((atomId_t)10);

  // check non-equality: less
  comparisonMap.erase((atomId_t)9);
  CPPUNIT_ASSERT( comparisonMap != fileChecker.InternalAtomBondMap );
}

/** Unit tests for CheckAgainstAdjacencyFile::ParseInExternalMap().
 *
 */
void CheckAgainstAdjacencyFileTest::ParseInExternalMapTest()
{
  std::stringstream input(adjacencyfile);
  World::getInstance().selectAllAtoms(NoAtoms());
  CheckAgainstAdjacencyFile fileChecker(input);
  fileChecker.CreateInternalMap(World::getInstance().beginAtomSelection(), World::getInstance().endAtomSelection());

  // check size (it's 8*2 + 2*1 = 18 keys)
  CPPUNIT_ASSERT_EQUAL( (size_t)0, fileChecker.InternalAtomBondMap.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)18, fileChecker.ExternalAtomBondMap.size() );

  // check equality
  CPPUNIT_ASSERT( comparisonMap.size() == fileChecker.ExternalAtomBondMap.size() );
  CPPUNIT_ASSERT( comparisonMap == fileChecker.ExternalAtomBondMap );

  // check non-equality: more
  comparisonMap.insert( std::make_pair( (atomId_t)10, (atomId_t)8) );
  CPPUNIT_ASSERT( comparisonMap != fileChecker.ExternalAtomBondMap );
  comparisonMap.erase((atomId_t)10);

  // check non-equality: less
  comparisonMap.erase((atomId_t)9);
  CPPUNIT_ASSERT( comparisonMap != fileChecker.ExternalAtomBondMap );
}

/** Unit tests for CheckAgainstAdjacencyFile::CompareInternalExternalMap().
 *
 */
void CheckAgainstAdjacencyFileTest::CompareInternalExternalMapTest()
{
  std::stringstream input(adjacencyfile);
  CheckAgainstAdjacencyFile fileChecker(input);

  // assert non-equality before parsing
  CPPUNIT_ASSERT( fileChecker.InternalAtomBondMap.size() != fileChecker.ExternalAtomBondMap.size() );
  CPPUNIT_ASSERT( fileChecker.InternalAtomBondMap != fileChecker.ExternalAtomBondMap );
  CPPUNIT_ASSERT( !fileChecker.CompareInternalExternalMap() );

  // parse
  World::getInstance().selectAllAtoms(AllAtoms());
  fileChecker.CreateInternalMap(World::getInstance().beginAtomSelection(), World::getInstance().endAtomSelection());

  // assert equality after parsing
  CPPUNIT_ASSERT_EQUAL( fileChecker.InternalAtomBondMap.size(), fileChecker.ExternalAtomBondMap.size() );
  CPPUNIT_ASSERT_EQUAL( fileChecker.InternalAtomBondMap, fileChecker.ExternalAtomBondMap );
  CPPUNIT_ASSERT( fileChecker.CompareInternalExternalMap() );
}

/** Unit tests for CheckAgainstAdjacencyFile::operator()().
 *
 */
void CheckAgainstAdjacencyFileTest::operatorTest()
{
  World::getInstance().selectAllAtoms(AllAtoms());
  {
    // parse right
    std::stringstream input(adjacencyfile);
    CheckAgainstAdjacencyFile fileChecker(input);
    CPPUNIT_ASSERT( fileChecker(World::getInstance().beginAtomSelection(), World::getInstance().endAtomSelection()) );
  }
  {
    // parse wrong1
    std::stringstream input(wrongadjacencyfile1);
    CheckAgainstAdjacencyFile fileChecker(input);
    CPPUNIT_ASSERT( !fileChecker(World::getInstance().beginAtomSelection(), World::getInstance().endAtomSelection()) );
  }
  {
    // parse wrong2 (there is no atom 10)
    std::stringstream input(wrongadjacencyfile2);
#ifndef NDEBUG
    std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
    CPPUNIT_ASSERT_THROW( CheckAgainstAdjacencyFile fileChecker(input), Assert::AssertionFailure );
#else
    CheckAgainstAdjacencyFile fileChecker(input);
    CPPUNIT_ASSERT( !fileChecker(World::getInstance().beginAtomSelection(), World::getInstance().endAtomSelection()) );
#endif
  }
}
