/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CenterInBoxAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include "boost/serialization/vector.hpp"

#include "CodePatterns/MemDebug.hpp"

#include <boost/shared_ptr.hpp>

#include "Box.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/MatrixContent.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Actions/WorldAction/CenterInBoxAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "CenterInBoxAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr WorldCenterInBoxAction::performCall() {
  // create undo state
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  const RealSpaceMatrix &matrix = World::getInstance().getDomain().getM();
  oa << matrix;
  std::vector< boost::shared_ptr<Vector> > OldPositions;
  std::vector<molecule*> AllMolecules = World::getInstance().getAllMolecules();
  for (std::vector<molecule*>::iterator MolRunner = AllMolecules.begin();
      MolRunner != AllMolecules.end();
      ++MolRunner) {
    for(molecule::const_iterator AtomRunner = (*MolRunner)->begin();
        AtomRunner != (*MolRunner)->end();
        ++AtomRunner) {
      OldPositions.push_back(
          boost::shared_ptr<Vector>(new Vector(
              (*AtomRunner)->getPosition()
              ))
          );
    }
  }

  // set new domain
  World::getInstance().setDomain(params.cell_size.get().getM());

  // center atoms
  for (std::vector<molecule*>::iterator MolRunner = AllMolecules.begin(); MolRunner != AllMolecules.end(); ++MolRunner) {
    (*MolRunner)->CenterInBox();
  }

  // give final box size
  LOG(0, "Box domain is now " << World::getInstance().getDomain().getM());

  // create undo state
  WorldCenterInBoxState *UndoState =
      new WorldCenterInBoxState(
          undostream.str(),
          OldPositions,
          params
          );

  return Action::state_ptr(UndoState);
}

Action::state_ptr WorldCenterInBoxAction::performUndo(Action::state_ptr _state) {
  WorldCenterInBoxState *state = assert_cast<WorldCenterInBoxState*>(_state.get());

  // restore domain
  RealSpaceMatrix matrix;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> matrix;
  World::getInstance().setDomain(matrix);

  // place atoms on old positions
  std::vector< boost::shared_ptr<Vector> >::const_iterator OldPositionsIter = state->OldPositions.begin();
  std::vector<molecule*> AllMolecules = World::getInstance().getAllMolecules();
  for (std::vector<molecule*>::iterator MolRunner = AllMolecules.begin();
      MolRunner != AllMolecules.end();
      ++MolRunner) {
    for(molecule::iterator AtomRunner = (*MolRunner)->begin();
        AtomRunner != (*MolRunner)->end();
        ++AtomRunner) {
      ASSERT(OldPositionsIter != state->OldPositions.end(),
          "WorldBoundInBoxAction::performUndo() - too few positions stored in UndoState.");
      (*AtomRunner)->setPosition(**(OldPositionsIter++));
    }
  }

  // give final box size
  LOG(0, "Box domain restored to " << World::getInstance().getDomain().getM());

  return Action::state_ptr(_state);
}

Action::state_ptr WorldCenterInBoxAction::performRedo(Action::state_ptr _state){
  WorldCenterInBoxState *state = assert_cast<WorldCenterInBoxState*>(_state.get());

  // set new domain
  World::getInstance().setDomain(state->params.cell_size.get().getM());

  // center atoms
  std::vector<molecule *> AllMolecules = World::getInstance().getAllMolecules();
  for (std::vector<molecule*>::iterator MolRunner = AllMolecules.begin(); MolRunner != AllMolecules.end(); ++MolRunner) {
    (*MolRunner)->CenterInBox();
  }

  // give final box size
  LOG(0, "Box domain is again " << World::getInstance().getDomain().getM());

  return Action::state_ptr(_state);
}

bool WorldCenterInBoxAction::canUndo() {
  return true;
}

bool WorldCenterInBoxAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
