/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ForceAnnealingAction.cpp
 *
 *  Created on: Aug 02, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Actions/UndoRedoHelpers.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomicInfo.hpp"
#include "Atom/AtomSet.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Dynamics/ForceAnnealing.hpp"
#include "molecule.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#include <vector>
#include <iostream>
#include <fstream>
#include <string>

#include "Actions/MoleculeAction/ForceAnnealingAction.hpp"

using namespace MoleCuilder;

enum VectorIndexType {
  PositionIndex=0,
  VelocityIndex=1,
  ForceIndex=2
};

// and construct the stuff
#include "ForceAnnealingAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeForceAnnealingAction::performCall() {
  AtomSetMixin<std::vector<atom *> > set(World::getInstance().getSelectedAtoms());
  if (set.empty()) {
    STATUS("No atoms selected.");
    return Action::failure;
  }
  // first, we need to sort the mixin according to their ids (as selected atoms are sorted
  // according to their arbitrary address in memory)
  set.sortByIds();

  // create undo state for all selected atoms (undo info)
  std::vector<AtomicInfo> UndoInfo;
  UndoInfo.reserve(set.size());
  {
    for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
        iter != World::getInstance().endAtomSelection();
        ++iter)
      UndoInfo.push_back(AtomicInfo(*(iter->second)));
  }

  size_t CurrentStep = WorldTime::getInstance().getTime();
  {
    // copy current time step to new one and and proceed on this one
    for (World::AtomSelectionIterator iter = World::getInstance().beginAtomSelection();
        iter != World::getInstance().endAtomSelection();
        ++iter) {
      atom * const Walker = iter->second;
      Walker->setPositionAtStep(CurrentStep+1,
          Walker->getPositionAtStep(CurrentStep));
      if (!params.forcesfile.get().string().empty()) {
        // don't use forces or velocities from old step
        Walker->setAtomicVelocityAtStep(CurrentStep+1, zeroVec);
        Walker->setAtomicForceAtStep(CurrentStep+1, zeroVec);
      } else {
        // force have already been calculated, hence copy them
        Walker->setAtomicVelocityAtStep(CurrentStep+1,
            Walker->getAtomicVelocityAtStep(CurrentStep));
        Walker->setAtomicForceAtStep(CurrentStep+1,
            Walker->getAtomicForceAtStep(CurrentStep));
      }
    }
    // increment to next time step: re-creates bond graph
    ++CurrentStep;
    World::getInstance().setTime(CurrentStep);
  }
  ForceAnnealing<std::vector<atom *> > optimizer(
      set,
      params.deltat.get(),
      true,
      params.steps.get(),
      params.MaxDistance.get(),
      params.DampingFactor.get());
  // parse forces into next step
  if (!params.forcesfile.get().string().empty()) {
    LOG(1, "Parsing forces file.");
    if (!optimizer.parseForcesFile(params.forcesfile.get().string().c_str(), CurrentStep))
      LOG(2, "File " << params.forcesfile.get() << " not found.");
    else
      LOG(2, "File " << params.forcesfile.get() << " found and parsed.");
  }

  // perform optimization step
  LOG(1, "Structural optimization.");
  optimizer(CurrentStep, 1, params.UseBondGraph.get());
  STATUS("Successfully optimized structure by one step.");

//  // increment to next time step
//  World::getInstance().setTime(CurrentStep+1);

  std::vector<AtomicInfo> RedoInfo;
  RedoInfo.reserve(set.size());
  {
    for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
        iter != World::getInstance().endAtomSelection();
        ++iter)
      RedoInfo.push_back(AtomicInfo(*(iter->second)));
  }
  MoleculeForceAnnealingState *UndoState =
    new MoleculeForceAnnealingState(UndoInfo, RedoInfo, params);

  return ActionState::ptr(UndoState);
}

ActionState::ptr MoleculeForceAnnealingAction::performUndo(ActionState::ptr _state) {
  MoleculeForceAnnealingState *state =
      assert_cast<MoleculeForceAnnealingState*>(_state.get());

  // set stored old state
  SetAtomsFromAtomicInfo(state->UndoInfo);

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeForceAnnealingAction::performRedo(ActionState::ptr _state){
  MoleculeForceAnnealingState *state =
      assert_cast<MoleculeForceAnnealingState*>(_state.get());

  // set stored new state
  SetAtomsFromAtomicInfo(state->RedoInfo);

  return ActionState::ptr(_state);
}

bool MoleculeForceAnnealingAction::canUndo() {
  return true;
}

bool MoleculeForceAnnealingAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
