/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * fragmentation_helpers.cpp
 *
 *  Created on: Oct 18, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "fragmentation_helpers.hpp"

#include <algorithm>
#include <sstream>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Fragmentation/AdaptivityMap.hpp"
#include "Fragmentation/AtomMask.hpp"
#include "Fragmentation/Graph.hpp"
#include "Fragmentation/KeySet.hpp"
#include "Helpers/defs.hpp"
#include "Helpers/helpers.hpp"
#include "molecule.hpp"

using namespace std;

/** print atom mask for debugging.
 * \param AtomMask defines true/false per global Atom::Nr to mask in/out each nuclear site, used to activate given number of site to increment order adaptively
 * \param AtomCount number of entries in \a *AtomMask
 */
void PrintAtomMask(const AtomMask_t &AtomMask, int AtomCount)
{
  if (DoLog(2)) {
    {
      // TODO: Truncate cleverly for large atomcount?
      std::stringstream output;
      output << "              ";
      for(int i=0;i<AtomCount;i++)
        output << (i % 10);
      LOG(2, output.str());
    }
    {
      std::stringstream output;
      output << "Atom mask is: ";
      for(int i=0;i<AtomCount;i++)
        output << AtomMask.printBit(i);
      LOG(2, output.str());
    }
  }
}

/** Combines all KeySets from all orders into single ones (with just unique entries).
 * \param &FragmentList list to fill
 * \param ***FragmentLowerOrdersList
 * \param &RootStack stack with all root candidates (unequal to each atom in complete molecule if adaptive scheme is applied)
 * \param *mol molecule with atoms and bonds
 */
int CombineAllOrderListIntoOne(Graph &FragmentList, std::vector<Graph*> *&FragmentLowerOrdersList, KeyStack &RootStack, molecule *mol)
{
  int RootNr = 0;
  int RootKeyNr = 0;
  int StartNr = 0;
  int counter = 0;
  int NumLevels = 0;
  atom *Walker = NULL;

  LOG(0, "Combining the lists of all orders per order and finally into a single one.");
  counter = FragmentList.size();

  StartNr = RootStack.back();
  do {
    RootKeyNr = RootStack.front();
    RootStack.pop_front();
    Walker = mol->FindAtom(RootKeyNr);
    NumLevels = Walker->AdaptiveOrder;
    ASSERT( NumLevels == FragmentLowerOrdersList[RootNr].size(),
        "CombineAllOrderListIntoOne() - differing size and NumLevels.");
    for(int i=0;i<NumLevels;i++) {
      if (FragmentLowerOrdersList[RootNr][i] != NULL) {
        FragmentList.InsertGraph((*FragmentLowerOrdersList[RootNr][i]), counter);
      }
    }
    RootStack.push_back(Walker->getNr());
    RootNr++;
  } while (RootKeyNr != StartNr);
  return counter;
};

/** Free's memory allocated for all KeySets from all orders.
 * \param *out output stream for debugging
 * \param **&FragmentLowerOrdersList
 * \param &RootStack stack with all root candidates (unequal to each atom in complete molecule if adaptive scheme is applied)
 * \param *mol molecule with atoms and bonds
 */
void FreeAllOrdersList(std::vector<Graph*> *&FragmentLowerOrdersList, KeyStack &RootStack, molecule *mol)
{
  LOG(1, "Free'ing the lists of all orders per order.");
  int RootNr = 0;
  int RootKeyNr = 0;
  int NumLevels = 0;
  atom *Walker = NULL;
  while (!RootStack.empty()) {
    RootKeyNr = RootStack.front();
    RootStack.pop_front();
    Walker = mol->FindAtom(RootKeyNr);
    NumLevels = Walker->AdaptiveOrder;
    for(int i=0;i<NumLevels;i++) {
      if (FragmentLowerOrdersList[RootNr][i] != NULL) {
        delete FragmentLowerOrdersList[RootNr][i];
      }
      FragmentLowerOrdersList[RootNr].clear();
    }
    RootNr++;
  }
  delete[] FragmentLowerOrdersList;
};

const std::vector<atomId_t> getGlobalIdsFromLocalIds(const molecule &mol, const std::vector<atomId_t> &atomids)
{
  std::vector<atomId_t> globalids;
  std::transform(atomids.begin(), atomids.end(), std::back_inserter(globalids),
      boost::bind( &atom::getId,
          boost::bind( &molecule::FindAtom, boost::cref(mol), _1 ))
  );
  return globalids;
}
