/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * InterfaceVMGDebugGridJob.cpp
 *
 *  Created on: 10.06.2012
 *      Author: Frederik Heber
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MPI
#include "mpi.h"
#endif

#include "base/vector.hpp"
#include "base/math.hpp"
#include "comm/comm.hpp"
#include "grid/grid.hpp"
#include "grid/multigrid.hpp"
#include "mg.hpp"

#include "InterfaceVMGDebugGridJob.hpp"

#include "CodePatterns/MemDebug.hpp"

#include <cmath>
#include <iostream>

#include "CodePatterns/Log.hpp"


using namespace VMG;
using VMGInterfaces::InterfaceVMGDebugGridJob;

InterfaceVMGDebugGridJob::InterfaceVMGDebugGridJob(
    const std::vector< double > &_sampled_input,
    VMG::Boundary boundary,
    int levelMin,
    int levelMax,
    const VMG::Vector &box_begin,
    vmg_float box_end,
    const int& near_field_cells,
    int coarseningSteps,
    double alpha) :
  VMG::Interface(boundary, levelMin, levelMax,
      box_begin, box_end, coarseningSteps, alpha),
  sampled_input(_sampled_input),
  level(levelMax)
{
}

InterfaceVMGDebugGridJob::~InterfaceVMGDebugGridJob()
{}

void InterfaceVMGDebugGridJob::ImportRightHandSide(Multigrid& multigrid)
{
  Index i;
  Vector pos;
  //  VMG::TempGrid *temp_grid = new VMG::TempGrid(129, 0, 0., 1.);

  Grid& grid = multigrid(multigrid.MaxLevel());
  //grid.ClearBoundary(); // we don't have a boundary under periodic boundary conditions

  // print debugging info on grid size
  LOG(1, "INFO: Mesh has extent " << grid.Extent().MeshWidth() << ".");
  const int gridpoints = pow(2, level);
  LOG(1, "INFO: gridpoints on finest level are " << gridpoints << ".");
  LOG(1, "INFO: "
      << "X in [" << grid.Local().Begin().X() << "," << grid.Local().End().X() << "],"
      << "Y in [" << grid.Local().Begin().Y() << "," << grid.Local().End().Y() << "],"
      << "Z in [" << grid.Local().Begin().Z() << "," << grid.Local().End().Z() << "].");

  // add sampled grid points to grid
  std::vector<double>::const_iterator sample_iter = sampled_input.begin();
  for (Grid::iterator iter = grid.Iterators().Local().Begin();
      iter != grid.Iterators().Local().End();
      ++iter)
    grid(*iter) = -1. * (*sample_iter++);
  assert( sample_iter == sampled_input.end() );

  // print electron grid to vtk
  Comm& comm = *MG::GetComm();
  comm.PrintGrid(grid, "Sampled Grid");
}

void InterfaceVMGDebugGridJob::ExportSolution(Grid& grid)
{
  // we don't solve anything, hence nothing to do here
}
