/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * atom_atominfo.cpp
 *
 *  Created on: Oct 19, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "periodentafel.hpp"
#include "World.hpp"
#include "element.hpp"
#include "atom_atominfo.hpp"

/** Constructor of class AtomInfo.
 */
AtomInfo::AtomInfo() :
  AtomicElement(NULL)
{};

/** Copy constructor of class AtomInfo.
 */
AtomInfo::AtomInfo(const AtomInfo &_atom) :
    AtomicPosition(_atom.AtomicPosition),
    AtomicElement(_atom.AtomicElement)
{};

AtomInfo::AtomInfo(const VectorInterface &_v) :
    AtomicPosition(_v.getPosition()),
    AtomicElement(NULL)
{};

/** Destructor of class AtomInfo.
 */
AtomInfo::~AtomInfo()
{
};

const element *AtomInfo::getType() const
{
  return AtomicElement;
}

const double& AtomInfo::operator[](size_t i) const
{
  return AtomicPosition[i];
}

const double& AtomInfo::at(size_t i) const
{
  return AtomicPosition.at(i);
}

void AtomInfo::set(size_t i, const double value)
{
  AtomicPosition.at(i) = value;
}

const Vector& AtomInfo::getPosition() const
{
  return AtomicPosition;
}

void AtomInfo::setType(const element* _type) {
  AtomicElement = _type;
}

void AtomInfo::setType(const int Z) {
  const element *elem = World::getInstance().getPeriode()->FindElement(Z);
  setType(elem);
}

void AtomInfo::setPosition(const Vector& _vector)
{
  AtomicPosition = _vector;
  //cout << "AtomInfo::setPosition: " << getType()->symbol << " at " << getPosition() << endl;
}

const VectorInterface& AtomInfo::operator+=(const Vector& b)
{
  AtomicPosition += b;
  return *this;
}

const VectorInterface& AtomInfo::operator-=(const Vector& b)
{
  AtomicPosition -= b;
  return *this;
}

Vector const AtomInfo::operator+(const Vector& b) const
{
  Vector a(AtomicPosition);
  a += b;
  return a;
}

Vector const AtomInfo::operator-(const Vector& b) const
{
  Vector a(AtomicPosition);
  a -= b;
  return a;
}

double AtomInfo::distance(const Vector &point) const
{
  return AtomicPosition.distance(point);
}

double AtomInfo::DistanceSquared(const Vector &y) const
{
  return AtomicPosition.DistanceSquared(y);
}

double AtomInfo::distance(const VectorInterface &_atom) const
{
  return _atom.distance(AtomicPosition);
}

double AtomInfo::DistanceSquared(const VectorInterface &_atom) const
{
  return _atom.DistanceSquared(AtomicPosition);
}

VectorInterface &AtomInfo::operator=(const Vector& _vector)
{
  AtomicPosition = _vector;
  return *this;
}

void AtomInfo::ScaleAll(const double *factor)
{
  AtomicPosition.ScaleAll(factor);
}

void AtomInfo::ScaleAll(const Vector &factor)
{
  AtomicPosition.ScaleAll(factor);
}

void AtomInfo::Scale(const double factor)
{
  AtomicPosition.Scale(factor);
}

void AtomInfo::Zero()
{
  AtomicPosition.Zero();
}

void AtomInfo::One(const double one)
{
  AtomicPosition.One(one);
}

void AtomInfo::LinearCombinationOfVectors(const Vector &x1, const Vector &x2, const Vector &x3, const double * const factors)
{
  AtomicPosition.LinearCombinationOfVectors(x1,x2,x3,factors);
}

const AtomInfo& operator*=(AtomInfo& a, const double m)
{
  a.Scale(m);
  return a;
}

AtomInfo const operator*(const AtomInfo& a, const double m)
{
  AtomInfo copy(a);
  copy *= m;
  return copy;
}

AtomInfo const operator*(const double m, const AtomInfo& a)
{
  AtomInfo copy(a);
  copy *= m;
  return copy;
}

std::ostream & AtomInfo::operator << (std::ostream &ost) const
{
  return (ost << getPosition());
}

std::ostream & operator << (std::ostream &ost, const AtomInfo &a)
{
  ost << a;
  return ost;
}

