/*
 * Parameter_impl.hpp
 *
 *  Created on: Apr 16, 2012
 *      Author: ankele
 */

#ifndef PARAMETER_IMPL_HPP_
#define PARAMETER_IMPL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Parameter.hpp"
#include "ParameterExceptions.hpp"


template<typename T>
Parameter<T>::Parameter(const Parameter<T> &instance) :
  ParameterInterface(instance.getName()),
  value(instance.value.getValidator())
{
  value.set(instance.value.getUnvalidated());
}

/** Constructor for class Parameter.
 *
 */
template<typename T>
Parameter<T>::Parameter() :
  ParameterInterface("__no_name__"),
  value()
{};

/** Constructor for class Parameter.
 *
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name) :
  ParameterInterface(_name),
  value()
{};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _value initial value to set
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const T &_value) :
  ParameterInterface(_name),
  value()
{
  value.set(_value);
};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const Validator<T> &_Validator) :
  ParameterInterface(_name),
  value(_Validator)
{};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 * @param _value initial value to set
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const Validator<T> &_Validator, const T &_value) :
  ParameterInterface(_name),
  value(_Validator)
{
  value.set(_value);
};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const std::vector<T> &_ValidValues) :
  ParameterInterface(_name),
  value(_ValidValues)
{};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 * @param _value initial value to set
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const std::vector<T> &_ValidValues, const T &_value) :
  ParameterInterface(_name),
  value(_ValidValues)
{
  value.set(_value);
};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const range<T> &_ValidRange) :
  ParameterInterface(_name),
  value(_ValidRange)
{};

/** Constructor for class Parameter.
 *
 * @param _name name of this parameter
 * @param _ValidRange valid range for this ContinuousValue
 * @param _value initial value to set
 */
template<typename T>
Parameter<T>::Parameter(const std::string &_name, const range<T> &_ValidRange, const T &_value) :
  ParameterInterface(_name),
  value(_ValidRange)
{
  value.set(_value);
};

/** Destructor for class Parameter.
 *
 */
template<typename T>
Parameter<T>::~Parameter()
{};

/** Catch call to value.isValidAsString() to add exception information.
 *
 * @param _value value to set to
 */
template<typename T>
inline
bool Parameter<T>::isValidAsString(const std::string &_value) const throw(ParameterValidatorException)
{
  try {
    return value.isValidAsString(_value);
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}


/** Catch call to value.getAsString() to add exception information.
 *
 * @return parameter value as string
 */
template<typename T>
inline const std::string Parameter<T>::getAsString() const throw(ParameterValueException)
{
  try {
    return value.getAsString();
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Catch call to value.getAsStringUnvalidated() to add exception information.
 *
 * @return parameter value as string
 */
template<typename T>
inline const std::string Parameter<T>::getAsStringUnvalidated() const throw(ParameterValueException)
{
  try {
    return value.getAsStringUnvalidated();
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Catch call to value.isValid() to add exception information.
 *
 * @return parameter value as string
 */
template<typename T>
inline bool Parameter<T>::isValid(const T &_value) const throw(ParameterValidatorException)
{
  try {
    return value.isValid(_value);
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}


/** Catch call to value.getUnvalidated() to add exception information.
 *
 * @return parameter value as string
 */
template<typename T>
inline const T & Parameter<T>::getUnvalidated() const throw(ParameterValueException)
{
  try {
    return value.getUnvalidated();
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Catch call to value.get() to add exception information.
 *
 * @return parameter value as string
 */
template<typename T>
inline const T & Parameter<T>::get() const throw(ParameterValueException)
{
  try {
    return value.get();
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Catch call to value.set() to add exception information.
 *
 * @param _value value to set to
 */
template<typename T>
inline void Parameter<T>::set(const T & _value) throw(ParameterValueException)
{
  try {
    value.set(_value);
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Catch call to value.set() to add exception information.
 *
 * @param _value value to set to
 */
template<typename T>
inline void Parameter<T>::setAsString(const std::string &_value) throw(ParameterValueException)
{
  try {
    value.setAsString(_value);
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
}

/** Compares this continuous value against another \a _instance.
 *
 * @param _instance other value to compare to
 * @return true - if contained ContinuousValue and name are the same, false - else
 */
template <class T>
bool Parameter<T>::operator==(const Parameter<T> &_instance) const throw(ParameterException)
{
  bool status = true;
  try {
    status = status &&
        (getUnvalidated() == _instance.getUnvalidated());
    status = status && (ParameterInterface::getName() == _instance.ParameterInterface::getName());
  } catch(ParameterException &e) {
    e << ParameterName(ParameterInterface::getName());
    throw;
  }
  return status;
}

/** Creates a clone of this Parameter instance.
 *
 * @return cloned instance
 */
template<typename T>
inline ParameterInterface* Parameter<T>::clone() const
{
  Parameter<T> *instance = new Parameter<T>(ParameterInterface::getName(), value.getValidator());
  // do not use get, we do not check for validity here
  if (value.isSet())
    instance->set(value.getUnvalidated());
  return instance;
}


#endif /* Parameter_IMPL_HPP_ */
