/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Psi3Parser_Parameters.cpp
 *
 *  Created on: Feb 3, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <string>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Psi3Parser.hpp"

#include "Psi3Parser_Parameters.hpp"

#include "Parser/Parameters/ContinuousParameter.hpp"
#include "Parser/Parameters/DiscreteParameter.hpp"
#include "Parser/Parameters/StringParameter.hpp"

// TODO: ContinuousValue<bool>::get() must be defined inline otherwise we get multiple definition of virtual thunk compilation errors
template <>
inline const std::string ContinuousValue<bool>::get() const
{
  ASSERT(ValueSet,
      "ContinuousValue<bool>::get() - requesting unset value.");
  if (value)
    return std::string("yes");
  else
    return std::string("no");
}

// TODO: ContinuousValue<bool>::set must be defined inline otherwise we get multiple definition of virtual thunk compilation errors
template <>
inline void ContinuousValue<bool>::set(const std::string _value)
{
  if (_value == std::string("yes")) {
    setValue(true);
  } else if (_value == std::string("no")) {
    setValue(false);
  } else {
    ASSERT(0,
        "void ContinuousValue<bool>::set() - value "+_value+" is neither yes or no.");
  }
}


Psi3Parser_Parameters::Psi3Parser_Parameters()
{
  Init();
}

void Psi3Parser_Parameters::Init()
{
  // add all known basis
  //initBasis();

  // add all parameter names
  {
    ParamNames.clear();
    ParamNames.resize(unknownParam);
    ParamNames[labelParam] = "label";
    ParamNames[jobtypeParam] = "jobtype";
    ParamNames[wavefunctionParam] = "wfn";
    ParamNames[maxiterParam] = "maxiter";
    ParamNames[referenceParam] = "reference";
    ParamNames[basisParam] = "basis";
    ParamNames[freeze_coreParam] = "freeze_core";
    ParamNames[unitsParam] = "units";
    ParamNames[dertypeParam] = "dertype";
    ParamNames[originParam] = "origin";
    ParamNames[multipParam] = "multip";
    ParamNames[chargeParam] = "charge";
    ParamNames[soccParam] = "socc";
    ParamNames[doccParam] = "docc";
    ParamNames[subgroupParam] = "subgroup";
    ParamNames[unique_axisParam] = "unique_axis";
  }

  // create freeze_core parameter
  {
    ValidFreezeCore.clear();
    ValidFreezeCore.resize(unknownFreezeCore);
    ValidFreezeCore[YES]="yes";
    ValidFreezeCore[TRUE]="true";
    ValidFreezeCore[NO]="no";
    ValidFreezeCore[FALSE]="false";
    ValidFreezeCore[SMALL]="small";
    ValidFreezeCore[LARGE]="large";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[freeze_coreParam],
            ValidFreezeCore,
            ValidFreezeCore[YES]));
  }

  // create units parameter
  {
    ValidUnits.clear();
    ValidUnits.resize(unknownUnits);
    ValidUnits[angstrom]="angstrom";
    ValidUnits[bohr]="bohr";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[unitsParam],
            ValidUnits,
            ValidUnits[angstrom]));
  }

  // create dertype parameter
  {
    ValidDerivativeType.clear();
    ValidDerivativeType.resize(unknownDerivativeType);
    ValidDerivativeType[NONE]="none";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[dertypeParam],
            ValidDerivativeType,
            ValidDerivativeType[NONE]));
  }

  // create unique_axis parameter
  {
    ValidUniqueAxis.clear();
    ValidUniqueAxis.resize(unknownUniqueAxis);
    ValidUniqueAxis[X]="x";
    ValidUniqueAxis[Y]="y";
    ValidUniqueAxis[Z]="z";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[unique_axisParam],
            ValidUniqueAxis,
            ValidUniqueAxis[X]));
  }

  // create jobtype parameter
  {
    ValidJobtypes.clear();
    ValidJobtypes.resize(unknownJobtype);
    ValidJobtypes[SP]="sp";
    ValidJobtypes[OPT]="opt";
    ValidJobtypes[DISP]="disp";
    ValidJobtypes[FREQ]="freq";
    ValidJobtypes[SYMM_FREQ]="symm_freq";
    ValidJobtypes[DBOC]="dboc";
    ValidJobtypes[RESPONSE]="response";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[jobtypeParam],
            ValidJobtypes,
            ValidJobtypes[SP]));
  }

  // create wavefunction parameter
  {
    ValidWavefunction.clear();
    ValidWavefunction.resize(unknownWavefunction);
    ValidWavefunction[SCF]="scf";
    ValidWavefunction[MP2]="mp2";
    ValidWavefunction[MP2R12]="mp2r12";
    ValidWavefunction[CIS]="cis";
    ValidWavefunction[DETCI]="detci";
    ValidWavefunction[CASSCF]="casscf";
    ValidWavefunction[RASSCF]="rasscf";
    ValidWavefunction[CCSD]="ccsd";
    ValidWavefunction[CCSD_T]="ccsd_t";
    ValidWavefunction[BCCD]="bccd";
    ValidWavefunction[BCCD_T]="bccd_t";
    ValidWavefunction[EOM_CCSD]="eom_ccsd";
    ValidWavefunction[ZAPTN]="zaptn";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[wavefunctionParam],
            ValidWavefunction,
            ValidWavefunction[SCF]));
  }

  // create reference parameter
  {
    ValidReference.clear();
    ValidReference.resize(unknownReference);
    ValidReference[RHF]="rhf";
    ValidReference[ROHF]="rohf";
    ValidReference[UHF]="uhf";
    ValidReference[TWOCON]="twocon";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[referenceParam],
            ValidReference,
            ValidReference[RHF]));
  }

  // add all continuous parameters
  {
    appendParameter(new StringParameter(ParamNames[labelParam], std::string("unknown job")));
    appendParameter(new ContinuousParameter<int>(ParamNames[maxiterParam], 80));
    appendParameter(new StringParameter(ParamNames[basisParam], std::string("cc-pVTZ")));
    appendParameter(new StringParameter(ParamNames[originParam], std::string("(0.0\t0.0\t0.0)"))); // TODO: this should be a vector
    appendParameter(new ContinuousParameter<int>(ParamNames[multipParam], 1));
    appendParameter(new ContinuousParameter<int>(ParamNames[chargeParam], 0));
    appendParameter(new StringParameter(ParamNames[soccParam], std::string("()")));
    appendParameter(new StringParameter(ParamNames[doccParam], std::string("()")));
    appendParameter(new StringParameter(ParamNames[subgroupParam], std::string("")));
 }
}

Psi3Parser_Parameters::~Psi3Parser_Parameters()
{}

/** Getter for a specific Parameter.
 *
 * @param param index among enum Parameters
 * @return value of the desired Parameters
 */
const std::string Psi3Parser_Parameters::getParameter(const enum Parameters param) const
{
  return FormatParser_Parameters::getParameter(ParamNames[param])->get();
}

/** Setter for a specific Parameter.
 *
 * @param param index among enum Parameters
 * @param _value value to set desired Parameter to
 */
void Psi3Parser_Parameters::setParameter(const enum Parameters param, const std::string &_value)
{
  const std::string &name = getParameterName(param);
  FormatParser_Parameters::getParameter(name)->set(_value);
}

/** Getter for name of a specific Parameter.
 *
 * @param param index among enum Parameters
 * @return name of the desired Parameter
 */
const std::string &Psi3Parser_Parameters::getParameterName(const enum Parameters param) const
{
  return ParamNames[param];
}

/** Checks whether all elements in the world also have parameters in the basis.
 *
 * @return true - all elements parametrized, false - at least one element is missing.
 */
bool Psi3Parser_Parameters::checkWorldElementsAgainstCurrentBasis() const
{
  DoeLog(0) && (eLog() << Verbose(0)
      << "Psi3Parser_Parameters::checkWorldElementsAgainstCurrentBasis() - not implemented yet."
      << std::endl);

  return false;
}

