/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * controller_SystemCommandJob.cpp
 *
 *  Created on: 01.06.2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "controller_SystemCommandJob.hpp"

#include <boost/assign.hpp>
#include <boost/bind.hpp>

#include "FragmentController.hpp"
#include "Jobs/SystemCommandJob.hpp"
#include "Results/FragmentResult.hpp"

#include "ControllerCommand.hpp"
#include "ControllerCommandRegistry.hpp"
#include "ControllerOptions_SystemCommandJob.hpp"

/** Print received results.
 *
 * @param results received results to print
 */
void printReceivedResults(const std::vector<FragmentResult::ptr> &results)
{
  for (std::vector<FragmentResult::ptr>::const_iterator iter = results.begin();
      iter != results.end(); ++iter)
    LOG(1, "RESULT: job #"+toString((*iter)->getId())+": "+toString((*iter)->result));
}

/** Creates a SystemCommandJob out of give \a command with \a argument.
 *
 * @param controller reference to controller to add jobs
 * @param ControllerInfo information on the job
 */
void createJobs(FragmentController &controller, const ControllerOptions_SystemCommandJob &ControllerInfo)
{
  const JobId_t next_id = controller.getAvailableId();
  LOG(2, "DEBUG: Creating new SystemCommandJob with '"
      +toString(ControllerInfo.executable)+"' and argument '"
      +toString(ControllerInfo.jobcommand)+"'.");
  FragmentJob::ptr testJob(
      new SystemCommandJob(ControllerInfo.executable, ControllerInfo.jobcommand, next_id) );
  std::vector<FragmentJob::ptr> jobs;
  jobs.push_back(testJob);
  controller.addJobs(jobs);
  controller.sendJobs(ControllerInfo.server, ControllerInfo.serverport);
  LOG(1, "INFO: Added one SystemCommandJob.");
}

inline std::vector<std::string> getListOfCommands(const ControllerCommandRegistry &ControllerCommands)
{
  std::vector<std::string> Commands;
  for (ControllerCommandRegistry::const_iterator iter = ControllerCommands.getBeginIter();
      iter != ControllerCommands.getEndIter(); ++iter)
    Commands.push_back(iter->first);

  return Commands;
}

ControllerOptions *controller_SystemCommandJob::allocateControllerInfo()
{
  return new ControllerOptions_SystemCommandJob();
}

void controller_SystemCommandJob::addSpecificCommands(
    boost::function<void (ControllerCommand *)> &registrator,
    FragmentController &controller,
    ControllerOptions &ControllerInfo)
{
  ControllerOptions_SystemCommandJob &CI =
      reinterpret_cast<ControllerOptions_SystemCommandJob &>(ControllerInfo);
  registrator(new ControllerCommand("createjobs",
      boost::assign::list_of< ControllerCommand::commands_t >
        (boost::bind(&FragmentController::requestIds,
            boost::ref(controller), boost::cref(ControllerInfo.server), boost::cref(ControllerInfo.serverport), 1))
        (boost::bind(&createJobs, boost::ref(controller), boost::cref(CI)))
      ));
  registrator(new ControllerCommand("receiveresults",
      boost::assign::list_of< ControllerCommand::commands_t >
        (boost::bind(&FragmentController::receiveResults,
            boost::ref(controller), boost::cref(ControllerInfo.server), boost::cref(ControllerInfo.serverport)))
        (boost::bind(&printReceivedResults,
            boost::bind(&FragmentController::getReceivedResults, boost::ref(controller))))
      ));
}

void controller_SystemCommandJob::addSpecificOption(
    boost::program_options::options_description_easy_init option)
{
  option
    ("jobcommand", boost::program_options::value< std::string >(), "command argument for executable for 'createjobs'")
    ("executable", boost::program_options::value< std::string >(), "executable for commands 'createjobs'")
    ;
}

int controller_SystemCommandJob::addOtherParsings(
    ControllerOptions &ControllerInfo,
    boost::program_options::variables_map &vm)
{
  ControllerOptions_SystemCommandJob &CI =
      reinterpret_cast<ControllerOptions_SystemCommandJob &>(ControllerInfo);
  int status = 0;
  status = CI.parseExecutable(vm);
  if (status) return status;
  status = CI.parseJobCommand(vm);
  return status;
}
